#pragma once
/*
 *      Copyright (C) 2005-2010 Team XBMC
 *      http://www.xbmc.org
 *
 *  This Program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This Program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with XBMC; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *  http://www.gnu.org/copyleft/gpl.html
 *
 */

namespace JSONRPC
{
  const char* const JSONRPC_SERVICE_ID          = "http://www.xbmc.org/jsonrpc/ServiceDescription.json";
  const int         JSONRPC_SERVICE_VERSION     = 3;
  const char* const JSONRPC_SERVICE_DESCRIPTION = "JSON RPC API of XBMC";

  const char* const JSONRPC_SERVICE_TYPES[] = {  
    "\"Optional.Boolean\": {"
      "\"type\": [ \"null\", \"boolean\" ],"
      "\"default\": null"
    "}",
    "\"Array.String\": {"
      "\"type\": \"array\","
      "\"items\": { \"type\": \"string\", \"minLength\": 1 }"
    "}",
    "\"Array.Integer\": {"
      "\"type\": \"array\","
      "\"items\": { \"type\": \"integer\" }"
    "}",
    "\"Global.Time\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"hours\": { \"type\": \"integer\", \"required\": true, \"minimum\": 0, \"maximum\": 23 },"
        "\"minutes\": { \"type\": \"integer\", \"required\": true, \"minimum\": 0, \"maximum\": 59 },"
        "\"seconds\": { \"type\": \"integer\", \"required\": true, \"minimum\": 0, \"maximum\": 59 },"
        "\"milliseconds\": { \"type\": \"integer\", \"required\": true, \"minimum\": 0, \"maximum\": 999 }"
      "},"
      "\"additionalProperties\": false"
    "}",
    "\"Configuration.Notifications\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"Player\": { \"type\": \"boolean\", \"required\": true },"
        "\"GUI\": { \"type\": \"boolean\", \"required\": true },"
        "\"System\": { \"type\": \"boolean\", \"required\": true },"
        "\"VideoLibrary\": { \"type\": \"boolean\", \"required\": true },"
        "\"AudioLibrary\": { \"type\": \"boolean\", \"required\": true },"
        "\"Other\": { \"type\": \"boolean\", \"required\": true }"
      "},"
      "\"additionalProperties\": false"
    "}",
    "\"Configuration\": {"
      "\"type\": \"object\", \"required\": true,"
      "\"properties\": {"
        "\"notifications\": { \"$ref\": \"Configuration.Notifications\", \"required\": true }"
      "}"
    "}",
    "\"Library.Id\": {"
      "\"type\": \"integer\","
      "\"default\": -1,"
      "\"minimum\": 1"
    "}",
    "\"Playlist.Id\": {"
      "\"type\": \"integer\","
      "\"minimum\": 0,"
      "\"maximum\": 2,"
      "\"default\": -1"
    "}",
    "\"Playlist.Type\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"unknown\", \"video\", \"audio\", \"picture\", \"mixed\" ]"
    "}",
    "\"Playlist.Property.Name\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"type\", \"size\" ]"
    "}",
    "\"Playlist.Property.Value\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"type\": { \"$ref\": \"Playlist.Type\" },"
        "\"size\": { \"type\": \"integer\", \"minimum\": 0 }"
      "}"
    "}",
    "\"Playlist.Position\": {"
      "\"type\": \"integer\","
      "\"minimum\": 0,"
      "\"default\": -1"
    "}",
    "\"Playlist.Item\": {"
      "\"type\": ["
        "{ \"type\": \"object\", \"properties\": { \"file\": { \"type\": \"string\", \"description\": \"Path to a file (not a directory) to be added to the playlist\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"directory\": { \"type\": \"string\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"movieid\": { \"$ref\": \"Library.Id\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"episodeid\": { \"$ref\": \"Library.Id\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"musicvideoid\": { \"$ref\": \"Library.Id\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"artistid\": { \"$ref\": \"Library.Id\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"albumid\": { \"$ref\": \"Library.Id\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"songid\": { \"$ref\": \"Library.Id\", \"required\": true } }, \"additionalProperties\": false },"
        "{ \"type\": \"object\", \"properties\": { \"genreid\": { \"$ref\": \"Library.Id\", \"required\": true, \"description\": \"Identification of a genre from the AudioLibrary\" } }, \"additionalProperties\": false }"
      "]"
    "}",
    "\"Player.Id\": {"
      "\"type\": \"integer\","
      "\"minimum\": 0,"
      "\"maximum\": 2,"
      "\"default\": -1"
    "}",
    "\"Player.Type\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"video\", \"audio\", \"picture\" ]"
    "}",
    "\"Player.Position.Percentage\": {"
      "\"type\": \"number\","
      "\"minimum\": 0.0,"
      "\"maximum\": 100.0"
    "}",
    "\"Player.Speed\": {"
      "\"type\": \"object\","
      "\"required\": true,"
      "\"properties\": {"
        "\"speed\": { \"type\": \"integer\" }"
      "}"
    "}",
    "\"Player.Repeat\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"off\", \"one\", \"all\" ]"
    "}",
    "\"Player.Audio.Stream\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"index\": { \"type\": \"integer\", \"minimum\": 0, \"required\": true },"
        "\"name\": { \"type\": \"string\", \"required\": true },"
        "\"language\": { \"type\": \"string\", \"required\": true }"
      "}"
    "}",
    "\"Player.Audio.Stream.Extended\": {"
      "\"extends\": \"Player.Audio.Stream\","
      "\"properties\": {"
        "\"codec\": { \"type\": \"string\", \"required\": true },"
        "\"bitrate\": { \"type\": \"integer\", \"required\": true },"
        "\"channels\": { \"type\": \"integer\", \"required\": true }"
      "}"
    "}",
    "\"Player.Subtitle\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"index\": { \"type\": \"integer\", \"minimum\": 0, \"required\": true },"
        "\"name\": { \"type\": \"string\", \"required\": true },"
        "\"language\": { \"type\": \"string\", \"required\": true }"
      "}"
    "}",
    "\"Player.Property.Name\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"type\", \"partymode\", \"speed\", \"time\", \"percentage\","
                "\"totaltime\", \"playlistid\", \"position\", \"repeat\", \"shuffled\","
                "\"canseek\", \"canchangespeed\", \"canmove\", \"canzoom\", \"canrotate\","
                "\"canshuffle\", \"canrepeat\", \"currentaudiostream\", \"audiostreams\","
                "\"subtitleenabled\", \"currentsubtitle\", \"subtitles\" ]"
    "}",
    "\"Player.Property.Value\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"type\": { \"$ref\": \"Player.Type\" },"
        "\"partymode\": { \"type\": \"boolean\" },"
        "\"speed\": { \"type\": \"integer\" },"
        "\"time\": { \"$ref\": \"Global.Time\" },"
        "\"percentage\": { \"$ref\": \"Player.Position.Percentage\" },"
        "\"totaltime\": { \"$ref\": \"Global.Time\" },"
        "\"playlistid\": { \"$ref\": \"Playlist.Id\" },"
        "\"position\": { \"$ref\": \"Playlist.Position\" },"
        "\"repeat\": { \"$ref\": \"Player.Repeat\" },"
        "\"shuffled\": { \"type\": \"boolean\" },"
        "\"canseek\": { \"type\": \"boolean\" },"
        "\"canchangespeed\": { \"type\": \"boolean\" },"
        "\"canmove\": { \"type\": \"boolean\" },"
        "\"canzoom\": { \"type\": \"boolean\" },"
        "\"canrotate\": { \"type\": \"boolean\" },"
        "\"canshuffle\": { \"type\": \"boolean\" },"
        "\"canrepeat\": { \"type\": \"boolean\" },"
        "\"currentaudiostream\": { \"$ref\": \"Player.Audio.Stream.Extended\" },"
        "\"audiostreams\": { \"type\": \"array\", \"items\": { \"$ref\": \"Player.Audio.Stream.Extended\" } },"
        "\"subtitleenabled\": { \"type\": \"boolean\" },"
        "\"currentsubtitle\": { \"$ref\": \"Player.Subtitle\" },"
        "\"subtitles\": { \"type\": \"array\", \"items\": { \"$ref\": \"Player.Subtitle\" } }"
      "}"
    "}",
    "\"Player.Notifications.Item.Type\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"unknown\", \"movie\", \"episode\", \"musicvideo\", \"song\" ]"
    "}",
    "\"Player.Notifications.Item\": {"
      "\"type\": ["
        "{ \"type\": \"object\", \"description\": \"An unknown item does not have any additional information.\","
          "\"properties\": {"
            "\"type\": { \"$ref\": \"Player.Notifications.Item.Type\", \"required\": true }"
          "}"
        "},"
        "{ \"type\": \"object\", \"description\": \"An item known to the database has an identification.\","
          "\"properties\": {"
            "\"type\": { \"$ref\": \"Player.Notifications.Item.Type\", \"required\": true },"
            "\"id\": { \"$ref\": \"Library.Id\", \"required\": true  }"
          "}"
        "},"
        "{ \"type\": \"object\", \"description\": \"A movie item has a title and may have a release year.\","
          "\"properties\": {"
            "\"type\": { \"$ref\": \"Player.Notifications.Item.Type\", \"required\": true },"
            "\"title\": { \"type\": \"string\", \"required\": true },"
            "\"year\": { \"type\": \"integer\" }"
          "}"
        "},"
        "{ \"type\": \"object\", \"description\": \"A tv episode has a title and may have an episode number, season number and the title of the show it belongs to.\","
          "\"properties\": {"
            "\"type\": { \"$ref\": \"Player.Notifications.Item.Type\", \"required\": true },"
            "\"title\": { \"type\": \"string\", \"required\": true },"
            "\"episode\": { \"type\": \"integer\" },"
            "\"season\": { \"type\": \"integer\" },"
            "\"showtitle\": { \"type\": \"string\" }"
          "}"
        "},"
        "{ \"type\": \"object\", \"description\": \"A music video has a title and may have an album and an artist.\","
          "\"properties\": {"
            "\"type\": { \"$ref\": \"Player.Notifications.Item.Type\", \"required\": true },"
            "\"title\": { \"type\": \"string\", \"required\": true },"
            "\"album\": { \"type\": \"string\" },"
            "\"artist\": { \"type\": \"string\" }"
          "}"
        "},"
        "{ \"type\": \"object\", \"description\": \"A song has a title and may have an album, an artist and a track number.\","
          "\"properties\": {"
            "\"type\": { \"$ref\": \"Player.Notifications.Item.Type\", \"required\": true },"
            "\"title\": { \"type\": \"string\", \"required\": true },"
            "\"album\": { \"type\": \"string\" },"
            "\"artist\": { \"type\": \"string\" },"
            "\"track\": { \"type\": \"integer\" }"
          "}"
        "}"
      "]"
    "}",
    "\"Player.Notifications.Player\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"playerid\": { \"$ref\": \"Player.Id\", \"required\": true },"
        "\"speed\": { \"type\": \"integer\" }"
      "}"
    "}",
    "\"Player.Notifications.Player.Seek\": {"
      "\"extends\": \"Player.Notifications.Player\","
      "\"properties\": {"
        "\"time\": { \"$ref\": \"Global.Time\" },"
        "\"seekoffset\": { \"$ref\": \"Global.Time\" }"
      "}"
    "}",
    "\"Player.Notifications.Data\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"item\": { \"$ref\": \"Player.Notifications.Item\", \"required\": true },"
        "\"player\": { \"$ref\": \"Player.Notifications.Player\", \"required\": true }"
      "}"
    "}",
    "\"Item.Fields.Base\": {"
      "\"type\": \"array\","
      "\"uniqueItems\": true,"
      "\"items\": { \"type\": \"string\" }"
    "}",
    "\"Item.Details.Base\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"label\": { \"type\": \"string\", \"required\": true }"
      "}"
    "}",
    "\"Media.Details.Base\": {"
      "\"extends\": \"Item.Details.Base\","
      "\"properties\": {"
        "\"fanart\": { \"type\": \"string\" },"
        "\"thumbnail\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Library.Fields.Genre\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\", \"enum\": [ \"title\", \"thumbnail\" ] }"
    "}",
    "\"Library.Details.Genre\": {"
      "\"extends\": \"Item.Details.Base\","
      "\"properties\": {"
        "\"genreid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"title\": { \"type\": \"string\" },"
        "\"thumbnail\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Audio.Fields.Artist\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"enum\": [ \"instrument\", \"style\", \"mood\", \"born\", \"formed\","
                  "\"description\", \"genre\", \"died\", \"disbanded\","
                  "\"yearsactive\", \"musicbrainzartistid\", \"fanart\","
                  "\"thumbnail\" ]"
      "}"
    "}",
    "\"Audio.Fields.Album\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"enum\": [ \"title\", \"description\", \"artist\", \"genre\","
                  "\"theme\", \"mood\", \"style\", \"type\", \"albumlabel\","
                  "\"rating\", \"year\", \"musicbrainzalbumid\","
                  "\"musicbrainzalbumartistid\", \"fanart\", \"thumbnail\","
                  "\"artistid\" ]"
      "}"
    "}",
    "\"Audio.Fields.Song\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"enum\": [ \"title\", \"artist\", \"albumartist\", \"genre\", \"year\","
                  "\"rating\", \"album\", \"track\", \"duration\", \"comment\","
                  "\"lyrics\", \"musicbrainztrackid\", \"musicbrainzartistid\","
                  "\"musicbrainzalbumid\", \"musicbrainzalbumartistid\","
                  "\"playcount\", \"fanart\", \"thumbnail\", \"file\", \"artistid\","
                  "\"albumid\" ]"
      "}"
    "}",
    "\"Audio.Details.Base\": {"
      "\"extends\": \"Media.Details.Base\","
      "\"properties\": {"
        "\"genre\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Audio.Details.Media\": {"
      "\"extends\": \"Audio.Details.Base\","
      "\"properties\": {"
        "\"title\": { \"type\": \"string\" },"
        "\"artist\": { \"type\": \"string\" },"
        "\"year\": { \"type\": \"integer\" },"
        "\"rating\": { \"type\": \"integer\" },"
        "\"musicbrainzalbumid\": { \"type\": \"string\" },"
        "\"musicbrainzalbumartistid\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Audio.Details.Artist\": {"
      "\"extends\": \"Audio.Details.Base\","
      "\"properties\": {"
        "\"artistid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"artist\": { \"type\": \"string\", \"required\": true },"
        "\"instrument\": { \"type\": \"string\" },"
        "\"style\": { \"type\": \"string\" },"
        "\"mood\": { \"type\": \"string\" },"
        "\"born\": { \"type\": \"string\" },"
        "\"formed\": { \"type\": \"string\" },"
        "\"description\": { \"type\": \"string\" },"
        "\"died\": { \"type\": \"string\" },"
        "\"disbanded\": { \"type\": \"string\" },"
        "\"yearsactive\": { \"type\": \"string\" },"
        "\"musicbrainzartistid\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Audio.Details.Album\": {"
      "\"extends\": \"Audio.Details.Media\","
      "\"properties\": {"
        "\"albumid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"description\": { \"type\": \"string\" },"
        "\"theme\": { \"type\": \"string\" },"
        "\"mood\": { \"type\": \"string\" },"
        "\"style\": { \"type\": \"string\" },"
        "\"type\": { \"type\": \"string\" },"
        "\"albumlabel\": { \"type\": \"string\" },"
        "\"artistid\": { \"$ref\": \"Library.Id\" }"
      "}"
    "}",
    "\"Audio.Details.Song\": {"
      "\"extends\": \"Audio.Details.Media\","
      "\"properties\": {"
        "\"songid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"file\": { \"type\": \"string\" },"
        "\"albumartist\": { \"type\": \"string\" },"
        "\"album\": { \"type\": \"string\" },"
        "\"track\": { \"type\": \"integer\" },"
        "\"duration\": { \"type\": \"integer\" },"
        "\"comment\": { \"type\": \"string\" },"
        "\"lyrics\": { \"type\": \"string\" },"
        "\"playcount\": { \"type\": \"integer\" },"
        "\"musicbrainztrackid\": { \"type\": \"string\" },"
        "\"musicbrainzartistid\": { \"type\": \"string\" },"
        "\"artistid\": { \"$ref\": \"Library.Id\" },"
        "\"albumid\": { \"$ref\": \"Library.Id\" }"
      "}"
    "}",
    "\"Video.Fields.Movie\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"description\": \"Requesting the cast, set, showling and/or resume field will result in increased response times\","
        "\"enum\": [ \"title\", \"genre\", \"year\", \"rating\", \"director\", \"trailer\","
                  "\"tagline\", \"plot\", \"plotoutline\", \"originaltitle\", \"lastplayed\","
                  "\"playcount\", \"writer\", \"studio\", \"mpaa\", \"cast\", \"country\","
                  "\"imdbnumber\", \"premiered\", \"productioncode\", \"runtime\", \"set\","
                  "\"showlink\", \"streamdetails\", \"top250\", \"votes\", \"fanart\","
                  "\"thumbnail\", \"file\", \"sorttitle\", \"resume\", \"setid\" ]"
      "}"
    "}",
    "\"Video.Fields.MovieSet\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"enum\": [ \"title\", \"playcount\", \"fanart\", \"thumbnail\" ]"
      "}"
    "}",
    "\"Video.Fields.TVShow\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"description\": \"Requesting the cast field will result in increased response times\","
        "\"enum\": [ \"title\", \"genre\", \"year\", \"rating\", \"plot\","
                  "\"studio\", \"mpaa\", \"cast\", \"playcount\", \"episode\","
                  "\"imdbnumber\", \"premiered\", \"votes\", \"lastplayed\","
                  "\"fanart\", \"thumbnail\", \"file\", \"originaltitle\","
                  "\"sorttitle\", \"episodeguide\" ]"
      "}"
    "}",
    "\"Video.Fields.Season\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"enum\": [ \"season\", \"showtitle\", \"playcount\", \"episode\", \"fanart\", \"thumbnail\", \"tvshowid\" ]"
      "}"
    "}",
    "\"Video.Fields.Episode\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"description\": \"Requesting the cast and/or resume field will result in increased response times\","
        "\"enum\": [ \"title\", \"plot\", \"votes\", \"rating\", \"writer\","
                  "\"firstaired\", \"playcount\", \"runtime\", \"director\","
                  "\"productioncode\", \"season\", \"episode\", \"originaltitle\","
                  "\"showtitle\", \"cast\", \"streamdetails\", \"lastplayed\", \"fanart\","
                  "\"thumbnail\", \"file\", \"resume\", \"tvshowid\" ]"
      "}"
    "}",
    "\"Video.Fields.MusicVideo\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"description\": \"Requesting the resume field will result in increased response times\","
        "\"enum\": [ \"title\", \"playcount\", \"runtime\", \"director\","
                  "\"studio\", \"year\", \"plot\", \"album\", \"artist\","
                  "\"genre\", \"track\", \"streamdetails\", \"lastplayed\","
                  "\"fanart\", \"thumbnail\", \"file\", \"resume\" ]"
      "}"
    "}",
    "\"Video.Cast\": {"
      "\"type\": \"array\","
      "\"items\": { \"type\": \"object\","
        "\"properties\": {"
          "\"name\": { \"type\": \"string\", \"required\": true },"
          "\"role\": { \"type\": \"string\", \"required\": true },"
          "\"thumbnail\": { \"type\": \"string\" }"
        "},"
        "\"additionalProperties\": false"
      "}"
    "}",
    "\"Video.Streams\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"audio\": { \"type\": \"array\", \"minItems\": 1,"
          "\"items\": { \"type\": \"object\","
            "\"properties\": {"
              "\"codec\": { \"type\": \"string\" },"
              "\"language\": { \"type\": \"string\" },"
              "\"channels\": { \"type\": \"integer\" }"
            "},"
            "\"additionalProperties\": false"
          "}"
        "},"
        "\"video\": { \"type\": \"array\", \"minItems\": 1,"
          "\"items\": { \"type\": \"object\","
            "\"properties\": {"
              "\"codec\": { \"type\": \"string\" },"
              "\"aspect\": { \"type\": \"number\" },"
              "\"width\": { \"type\": \"integer\" },"
              "\"height\": { \"type\": \"integer\" },"
              "\"duration\": { \"type\": \"integer\" }"
            "},"
            "\"additionalProperties\": false"
          "}"
        "},"
        "\"subtitle\": { \"type\": \"array\", \"minItems\": 1,"
          "\"items\": { \"type\": \"object\","
            "\"properties\": {"
              "\"language\": { \"type\": \"string\" }"
            "},"
            "\"additionalProperties\": false"
          "}"
        "}"
      "},"
      "\"additionalProperties\": false"
    "}",
    "\"Video.Resume\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"position\": { \"type\": \"number\", \"minimum\": 0 },"
        "\"total\": { \"type\": \"number\", \"minimum\": 0 }"
      "},"
      "\"additionalProperties\": false"
    "}",
    "\"Video.Details.Base\": {"
      "\"extends\": \"Media.Details.Base\","
      "\"properties\": {"
        "\"playcount\": { \"type\": \"integer\" }"
      "}"
    "}",
    "\"Video.Details.Media\": {"
      "\"extends\": \"Video.Details.Base\","
      "\"properties\": {"
        "\"title\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Video.Details.Item\": {"
      "\"extends\": \"Video.Details.Media\","
      "\"properties\": {"
        "\"file\": { \"type\": \"string\" },"
        "\"plot\": { \"type\": \"string\" },"
        "\"lastplayed\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Video.Details.File\": {"
      "\"extends\": \"Video.Details.Item\","
      "\"properties\": {"
        "\"runtime\": { \"type\": \"string\" },"
        "\"director\": { \"type\": \"string\" },"
        "\"streamdetails\": { \"$ref\": \"Video.Streams\" },"
        "\"resume\": { \"$ref\": \"Video.Resume\" }"
      "}"
    "}",
    "\"Video.Details.Movie\": {"
      "\"extends\": \"Video.Details.File\","
      "\"properties\": {"
        "\"movieid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"genre\": { \"type\": \"string\" },"
        "\"year\": { \"type\": \"integer\" },"
        "\"rating\": { \"type\": \"number\" },"
        "\"trailer\": { \"type\": \"string\" },"
        "\"tagline\": { \"type\": \"string\" },"
        "\"plotoutline\": { \"type\": \"string\" },"
        "\"originaltitle\": { \"type\": \"string\" },"
        "\"sorttitle\": { \"type\": \"string\" },"
        "\"writer\": { \"type\": \"string\" },"
        "\"studio\": { \"type\": \"string\" },"
        "\"mpaa\": { \"type\": \"string\" },"
        "\"cast\": { \"$ref\": \"Video.Cast\" },"
        "\"country\": { \"type\": \"string\" },"
        "\"imdbnumber\": { \"type\": \"string\" },"
        "\"premiered\": { \"type\": \"string\" },"
        "\"productioncode\": { \"type\": \"string\" },"
        "\"set\": { \"$ref\": \"Array.String\" },"
        "\"showlink\": { \"type\": \"string\" },"
        "\"top250\": { \"type\": \"integer\" },"
        "\"votes\": { \"type\": \"string\" },"
        "\"setid\": { \"$ref\": \"Array.Integer\" }"
      "}"
    "}",
    "\"Video.Details.MovieSet\": {"
      "\"extends\": \"Video.Details.Media\","
      "\"properties\": {"
        "\"setid\": { \"$ref\": \"Library.Id\", \"required\": true }"
      "}"
    "}",
    "\"Video.Details.MovieSet.Extended\": {"
      "\"extends\": \"Video.Details.MovieSet\","
      "\"properties\": {"
        "\"movies\": { \"type\": \"array\","
          "\"items\": { \"$ref\": \"Video.Details.Movie\" }"
        "}"
      "}"
    "}",
    "\"Video.Details.TVShow\": {"
      "\"extends\": \"Video.Details.Item\","
      "\"properties\": {"
        "\"tvshowid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"genre\": { \"type\": \"string\" },"
        "\"year\": { \"type\": \"integer\" },"
        "\"rating\": { \"type\": \"number\" },"
        "\"originaltitle\": { \"type\": \"string\" },"
        "\"sorttitle\": { \"type\": \"string\" },"
        "\"studio\": { \"type\": \"string\" },"
        "\"mpaa\": { \"type\": \"string\" },"
        "\"cast\": { \"$ref\": \"Video.Cast\" },"
        "\"episode\": { \"type\": \"integer\" },"
        "\"imdbnumber\": { \"type\": \"string\" },"
        "\"premiered\": { \"type\": \"string\" },"
        "\"votes\": { \"type\": \"string\" },"
        "\"episodeguide\": { \"type\": \"string\" }"
      "}"
    "}",
    "\"Video.Details.Season\": {"
      "\"extends\": \"Video.Details.Base\","
      "\"properties\": {"
        "\"season\": { \"type\": \"integer\", \"required\": true },"
        "\"showtitle\": { \"type\": \"string\" },"
        "\"episode\": { \"type\": \"integer\" },"
        "\"tvshowid\": { \"$ref\": \"Library.Id\" }"
      "}"
    "}",
    "\"Video.Details.Episode\": {"
      "\"extends\": \"Video.Details.File\","
      "\"properties\": {"
        "\"episodeid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"votes\": { \"type\": \"string\" },"
        "\"rating\": { \"type\": \"number\" },"
        "\"writer\": { \"type\": \"string\" },"
        "\"firstaired\": { \"type\": \"string\" },"
        "\"productioncode\": { \"type\": \"string\" },"
        "\"season\": { \"type\": \"integer\" },"
        "\"episode\": { \"type\": \"integer\" },"
        "\"originaltitle\": { \"type\": \"string\" },"
        "\"showtitle\": { \"type\": \"string\" },"
        "\"cast\": { \"$ref\": \"Video.Cast\" },"
        "\"tvshowid\": { \"$ref\": \"Library.Id\" }"
      "}"
    "}",
    "\"Video.Details.MusicVideo\": {"
      "\"extends\": \"Video.Details.File\","
      "\"properties\": {"
        "\"musicvideoid\": { \"$ref\": \"Library.Id\", \"required\": true },"
        "\"studio\": { \"type\": \"string\" },"
        "\"year\": { \"type\": \"integer\" },"
        "\"album\": { \"type\": \"string\" },"
        "\"artist\": { \"type\": \"string\" },"
        "\"genre\": { \"type\": \"string\" },"
        "\"track\": { \"type\": \"integer\" }"
      "}"
    "}",
    "\"Files.Media\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"video\", \"music\", \"pictures\", \"files\", \"programs\" ]"
    "}",
    "\"List.Amount\": {"
      "\"type\": \"integer\","
      "\"default\": -1,"
      "\"minimum\": 0"
    "}",
    "\"List.Limits\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"start\": { \"type\": \"integer\", \"minimum\": 0, \"default\": 0 },"
        "\"end\": { \"type\": \"integer\", \"minimum\": 0, \"default\": -1 }"
      "},"
      "\"additionalProperties\": false"
    "}",
    "\"List.Sort\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"method\": { \"type\": \"string\", \"default\": \"none\","
          "\"enum\": [ \"none\", \"label\", \"date\", \"size\", \"file\", \"drivetype\", \"track\", \"duration\", \"title\", \"artist\","
                    "\"album\", \"genre\", \"year\", \"videorating\", \"programcount\", \"playlist\", \"episode\", \"videotitle\","
                    "\"sorttitle\", \"productioncode\", \"songrating\", \"mpaarating\", \"videoruntime\", \"studio\", \"fullpath\","
                    "\"lastplayed\", \"unsorted\", \"max\" ]"
        "},"
        "\"order\": { \"type\": \"string\", \"default\": \"ascending\", \"enum\": [ \"ascending\", \"descending\" ] },"
        "\"ignorearticle\": { \"type\": \"boolean\", \"default\": false }"
      "}"
    "}",
    "\"List.LimitsReturned\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"start\": { \"type\": \"integer\", \"minimum\": 0, \"default\": 0 },"
        "\"end\": { \"type\": \"integer\", \"minimum\": 0, \"default\": -1 },"
        "\"total\": { \"type\": \"integer\", \"minimum\": 0, \"required\": true }"
      "},"
      "\"additionalProperties\": false"
    "}",
    "\"List.Fields.All\": {"
      "\"extends\": \"Item.Fields.Base\","
      "\"items\": { \"type\": \"string\","
        "\"enum\": [ \"title\", \"artist\", \"albumartist\", \"genre\", \"year\", \"rating\","
                  "\"album\", \"track\", \"duration\", \"comment\", \"lyrics\", \"musicbrainztrackid\","
                  "\"musicbrainzartistid\", \"musicbrainzalbumid\", \"musicbrainzalbumartistid\","
                  "\"playcount\", \"fanart\", \"director\", \"trailer\", \"tagline\", \"plot\","
                  "\"plotoutline\", \"originaltitle\", \"lastplayed\", \"writer\", \"studio\","
                  "\"mpaa\", \"cast\", \"country\", \"imdbnumber\", \"premiered\", \"productioncode\","
                  "\"runtime\", \"set\", \"showlink\", \"streamdetails\", \"top250\", \"votes\","
                  "\"firstaired\", \"season\", \"episode\", \"showtitle\", \"thumbnail\", \"file\","
                  "\"resume\", \"artistid\", \"albumid\", \"tvshowid\", \"setid\" ]"
      "}"
    "}",
    "\"List.Item.All\": {"
      "\"extends\": [ \"Video.Details.File\", \"Audio.Details.Media\" ],"
      "\"properties\": {"
        "\"id\": { \"$ref\": \"Library.Id\" },"
        "\"type\": { \"type\": \"string\", \"enum\": [ \"unknown\", \"movie\", \"episode\", \"musicvideo\", \"song\", \"picture\" ] },"
        "\"albumartist\": { \"type\": \"string\" },"
        "\"album\": { \"type\": \"string\" },"
        "\"track\": { \"type\": \"integer\" },"
        "\"duration\": { \"type\": \"integer\" },"
        "\"comment\": { \"type\": \"string\" },"
        "\"lyrics\": { \"type\": \"string\" },"
        "\"musicbrainztrackid\": { \"type\": \"string\" },"
        "\"musicbrainzartistid\": { \"type\": \"string\" },"
        "\"trailer\": { \"type\": \"string\" },"
        "\"tagline\": { \"type\": \"string\" },"
        "\"plotoutline\": { \"type\": \"string\" },"
        "\"originaltitle\": { \"type\": \"string\" },"
        "\"writer\": { \"type\": \"string\" },"
        "\"studio\": { \"type\": \"string\" },"
        "\"mpaa\": { \"type\": \"string\" },"
        "\"cast\": { \"$ref\": \"Video.Cast\" },"
        "\"country\": { \"type\": \"string\" },"
        "\"imdbnumber\": { \"type\": \"string\" },"
        "\"premiered\": { \"type\": \"string\" },"
        "\"productioncode\": { \"type\": \"string\" },"
        "\"set\": { \"$ref\": \"Array.String\" },"
        "\"showlink\": { \"type\": \"string\" },"
        "\"top250\": { \"type\": \"integer\" },"
        "\"votes\": { \"type\": \"string\" },"
        "\"firstaired\": { \"type\": \"string\" },"
        "\"season\": { \"type\": \"integer\" },"
        "\"episode\": { \"type\": \"integer\" },"
        "\"showtitle\": { \"type\": \"string\" },"
        "\"artistid\": { \"$ref\": \"Library.Id\" },"
        "\"albumid\": { \"$ref\": \"Library.Id\" },"
        "\"setid\": { \"$ref\": \"Array.Integer\" },"
        "\"tvshowid\": { \"$ref\": \"Library.Id\" }"
      "}"
    "}",
    "\"List.Item.File\": {"
      "\"extends\": \"List.Item.All\","
      "\"properties\": {"
        "\"file\": { \"type\": \"string\", \"required\": true },"
        "\"filetype\": { \"type\": \"string\", \"enum\": [ \"file\", \"directory\" ], \"required\": true }"
      "}"
    "}",
    "\"List.Items.Sources\": {"
      "\"type\": \"array\","
      "\"items\": {"
        "\"extends\": \"Item.Details.Base\","
        "\"properties\": {"
          "\"file\": { \"type\": \"string\", \"required\": true }"
        "}"
      "}"
    "}",
    "\"System.Property.Name\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"canshutdown\", \"cansuspend\", \"canhibernate\", \"canreboot\" ]"
    "}",
    "\"System.Property.Value\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"canshutdown\": { \"type\": \"boolean\" },"
        "\"cansuspend\": { \"type\": \"boolean\" },"
        "\"canhibernate\": { \"type\": \"boolean\" },"
        "\"canreboot\": { \"type\": \"boolean\" }"
      "}"
    "}",
    "\"Application.Property.Name\": {"
      "\"type\": \"string\","
      "\"enum\": [ \"volume\", \"muted\", \"name\", \"version\" ]"
    "}",
    "\"Application.Property.Value\": {"
      "\"type\": \"object\","
      "\"properties\": {"
        "\"volume\": { \"type\": \"integer\" },"
        "\"muted\": { \"type\": \"boolean\" },"
        "\"name\": { \"type\": \"string\", \"minLength\": 1 },"
        "\"version\": { \"type\": \"object\","
          "\"properties\": {"
            "\"major\": { \"type\": \"integer\", \"minimum\": 0, \"required\": true },"
            "\"minor\": { \"type\": \"integer\", \"minimum\": 0, \"required\": true },"
            "\"revision\": { \"type\": [ \"string\", \"integer\" ] },"
            "\"tag\": { \"type\": \"string\", \"enum\": [ \"prealpha\", \"alpha\", \"beta\", \"releasecandidate\", \"stable\" ], \"required\": true }"
          "}"
        "}"
      "}"
    "}"
  };

  const char* const JSONRPC_SERVICE_METHODS[] = {  
    "\"JSONRPC.Introspect\": {"
      "\"type\": \"method\","
      "\"description\": \"Enumerates all actions and descriptions\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"getdescriptions\", \"type\": \"boolean\", \"default\": true },"
        "{ \"name\": \"getmetadata\", \"type\": \"boolean\", \"default\": false },"
        "{ \"name\": \"filterbytransport\", \"type\": \"boolean\", \"default\": true },"
        "{ \"name\": \"filter\", \"type\": \"object\","
          "\"properties\": {"
            "\"id\": { \"type\": \"string\", \"required\": true, \"description\": \"Name of a namespace, method or type\" },"
            "\"type\": { \"type\": \"string\", \"required\": true, \"enum\": [ \"method\", \"namespace\", \"type\", \"notification\" ], \"description\": \"Type of the given name\" },"
            "\"getreferences\": { \"type\": \"boolean\", \"default\": true, \"description\": \"Whether or not to print the schema for referenced types\" }"
          "}"
        "}"
      "],"
      "\"returns\": \"object\""
    "}",
    "\"JSONRPC.Version\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve the jsonrpc protocol version\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"JSONRPC.Permission\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve the clients permissions\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": [],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"ReadData\": { \"type\": \"boolean\", \"required\": true },"
          "\"ControlPlayback\": { \"type\": \"boolean\", \"required\": true },"
          "\"ControlNotify\": { \"type\": \"boolean\", \"required\": true },"
          "\"ControlPower\": { \"type\": \"boolean\", \"required\": true },"
          "\"UpdateData\": { \"type\": \"boolean\", \"required\": true },"
          "\"RemoveData\": { \"type\": \"boolean\", \"required\": true },"
          "\"Navigate\": { \"type\": \"boolean\", \"required\": true },"
          "\"WriteFile\": { \"type\": \"boolean\", \"required\": true }"
        "}"
      "}"
    "}",
    "\"JSONRPC.Ping\": {"
      "\"type\": \"method\","
      "\"description\": \"Ping responder\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"JSONRPC.GetConfiguration\": {"
      "\"type\": \"method\","
      "\"description\": \"Get client-specific configurations\","
      "\"transport\": \"Announcing\","
      "\"permission\": \"ReadData\","
      "\"params\": [],"
      "\"returns\": { \"$ref\": \"Configuration\" }"
    "}",
    "\"JSONRPC.SetConfiguration\": {"
      "\"type\": \"method\","
      "\"description\": \"Change the client-specific configuration\","
      "\"transport\": \"Announcing\","
      "\"permission\": \"ControlNotify\","
      "\"params\": ["
        "{ \"name\": \"notifications\", \"type\": \"object\","
          "\"properties\": {"
            "\"Player\": { \"$ref\": \"Optional.Boolean\" },"
            "\"GUI\": { \"$ref\": \"Optional.Boolean\" },"
            "\"System\": { \"$ref\": \"Optional.Boolean\" },"
            "\"AudioLibrary\": { \"$ref\": \"Optional.Boolean\" },"
            "\"VideoLibrary\": { \"$ref\": \"Optional.Boolean\" },"
            "\"Other\": { \"$ref\": \"Optional.Boolean\" }"
          "}"
        "}"
      "],"
      "\"returns\": { \"$ref\": \"Configuration\" }"
    "}",
    "\"JSONRPC.NotifyAll\": {"
      "\"type\": \"method\","
      "\"description\": \"Notify all other connected clients\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"message\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"any\", \"default\": null }"
      "],"
      "\"returns\": \"any\""
    "}",
    "\"Player.Open\": {"
      "\"type\": \"method\","
      "\"description\": \"Start playback of either the playlist with the given ID, a slideshow with the pictures from the given directory or a single file or an item from the database.\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"item\", "
          "\"type\": ["
            "{ \"type\": \"object\", \"required\": true, \"additionalProperties\": false,"
              "\"properties\": {"
                "\"playlistid\": { \"$ref\": \"Playlist.Id\", \"required\": true },"
                "\"position\": { \"$ref\": \"Playlist.Position\", \"default\": 0 }"
              "}"
            "},"
            "{ \"$ref\": \"Playlist.Item\", \"required\": true },"
            "{ \"type\": \"object\", \"required\": true, \"additionalProperties\": false,"
              "\"properties\": {"
                "\"path\": { \"type\": \"string\", \"required\": true },"
                "\"random\": { \"type\": \"boolean\", \"default\": true },"
                "\"recursive\": { \"type\": \"boolean\", \"default\": true }"
              "}"
            "}"
          "]"
        "}"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.GetActivePlayers\": {"
      "\"type\": \"method\","
      "\"description\": \"Returns all active players\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": [],"
      "\"returns\": {"
        "\"type\": \"array\","
        "\"uniqueItems\": true,"
        "\"items\": { "
          "\"type\": \"object\","
          "\"properties\": {"
            "\"playerid\": { \"$ref\": \"Player.Id\", \"required\": true },"
            "\"type\": { \"$ref\": \"Player.Type\", \"required\": true }"
          "}"
        "}"
      "}"
    "}",
    "\"Player.GetProperties\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieves the values of the given properties\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"type\": \"array\", \"uniqueItems\": true, \"required\": true, \"items\": { \"$ref\": \"Player.Property.Name\" } }"
      "],"
      "\"returns\":  { \"$ref\": \"Player.Property.Value\", \"required\": true }"
    "}",
    "\"Player.GetItem\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieves the currently played item\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"List.Fields.All\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"item\": { \"$ref\": \"List.Item.All\", \"required\": true }"
        "}"
      "}"
    "}",
    "\"Player.PlayPause\": {"
      "\"type\": \"method\","
      "\"description\": \"Pauses or unpause playback and returns the new state\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": { \"$ref\": \"Player.Speed\" }"
    "}",
    "\"Player.Stop\": {"
      "\"type\": \"method\","
      "\"description\": \"Stops playback\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.SetSpeed\": {"
      "\"type\": \"method\","
      "\"description\": \"Set the speed of the current playback\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"speed\", \"type\": [ \"integer\", \"string\" ], \"enum\": [ -32, -16, -8, -4, -2, -1, 0, 1, 2, 4, 8, 16, 32, \"increment\", \"decrement\" ], \"required\": true }"
      "],"
      "\"returns\": { \"$ref\": \"Player.Speed\" }"
    "}",
    "\"Player.Seek\": {"
      "\"type\": \"method\","
      "\"description\": \"Seek through the playing item\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"value\", \"required\": true, \"type\": ["
            "{ \"$ref\": \"Player.Position.Percentage\", \"required\": true, \"description\": \"Percentage value to seek to\" },"
            "{ \"type\": \"object\", \"additionalProperties\": false, \"required\": true, \"description\": \"Time to seek to\","
              "\"properties\": {"
                "\"hours\": { \"type\": \"integer\", \"minimum\": 0, \"maximum\": 23 },"
                "\"minutes\": { \"type\": \"integer\", \"minimum\": 0, \"maximum\": 59 },"
                "\"seconds\": { \"type\": \"integer\", \"minimum\": 0, \"maximum\": 59 },"
                "\"milliseconds\": { \"type\": \"integer\", \"minimum\": 0, \"maximum\": 999 }"
              "}"
            "},"
            "{ \"type\": \"string\", \"enum\": [ \"smallforward\", \"smallbackward\", \"bigforward\", \"bigbackward\" ], \"required\": true, \"description\": \"Seek by predefined jumps\" }"
          "]"
        "}"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"percentage\": { \"$ref\": \"Player.Position.Percentage\" },"
          "\"time\": { \"$ref\": \"Global.Time\" },"
          "\"totaltime\": { \"$ref\": \"Global.Time\" }"
        "}"
      "}"
    "}",
    "\"Player.MoveLeft\": {"
      "\"type\": \"method\","
      "\"description\": \"If picture is zoomed move viewport left otherwise skip previous\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.MoveRight\": {"
      "\"type\": \"method\","
      "\"description\": \"If picture is zoomed move viewport right otherwise skip next\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.MoveDown\": {"
      "\"type\": \"method\","
      "\"description\": \"If picture is zoomed move viewport down\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.MoveUp\": {"
      "\"type\": \"method\","
      "\"description\": \"If picture is zoomed move viewport up\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.ZoomOut\": {"
      "\"type\": \"method\","
      "\"description\": \"Zoom out once\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.ZoomIn\": {"
      "\"type\": \"method\","
      "\"description\": \"Zoom in once\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.Zoom\": {"
      "\"type\": \"method\","
      "\"description\": \"Zooms current picture\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"value\", \"type\": \"integer\", \"required\": true, \"minimum\": 1, \"maximum\": 10, \"description\": \"Zoom level\" }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.Rotate\": {"
      "\"type\": \"method\","
      "\"description\": \"Rotates current picture\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.GoPrevious\": {"
      "\"type\": \"method\","
      "\"description\": \"Go to previous item on the playlist\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.GoNext\": {"
      "\"type\": \"method\","
      "\"description\": \"Go to next item on the playlist\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.GoTo\": {"
      "\"type\": \"method\","
      "\"description\": \"Go to item at the given position in the playlist\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"position\", \"$ref\": \"Playlist.Position\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.Shuffle\": {"
      "\"type\": \"method\","
      "\"description\": \"Shuffle items in the player\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.UnShuffle\": {"
      "\"type\": \"method\","
      "\"description\": \"Unshuffle items in the player\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.Repeat\": {"
      "\"type\": \"method\","
      "\"description\": \"Set the repeat mode of the player\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"state\", \"$ref\": \"Player.Repeat\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.SetAudioStream\": {"
      "\"type\": \"method\","
      "\"description\": \"Set the audio stream played by the player\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"stream\", \"required\": true, \"type\": ["
            "{ \"type\": \"string\", \"enum\": [ \"previous\", \"next\" ] },"
            "{ \"type\": \"integer\", \"minimum\": 0, \"description\": \"Index of the audio stream to play\" }"
          "]"
        "}"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Player.SetSubtitle\": {"
      "\"type\": \"method\","
      "\"description\": \"Set the subtitle displayed by the player\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playerid\", \"$ref\": \"Player.Id\", \"required\": true },"
        "{ \"name\": \"subtitle\", \"required\": true, \"type\": ["
            "{ \"type\": \"string\", \"enum\": [ \"previous\", \"next\", \"off\", \"on\" ] },"
            "{ \"type\": \"integer\", \"minimum\": 0, \"description\": \"Index of the subtitle to display\" }"
          "]"
        "}"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Playlist.GetPlaylists\": {"
      "\"type\": \"method\","
      "\"description\": \"Returns all existing playlists\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": [],"
      "\"returns\": {"
        "\"type\": \"array\","
        "\"uniqueItems\": true,"
        "\"items\": { "
          "\"type\": \"object\","
          "\"properties\": {"
            "\"playlistid\": { \"$ref\": \"Playlist.Id\", \"required\": true },"
            "\"type\": { \"$ref\": \"Playlist.Type\", \"required\": true }"
          "}"
        "}"
      "}"
    "}",
    "\"Playlist.GetProperties\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieves the values of the given properties\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"playlistid\", \"$ref\": \"Playlist.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"type\": \"array\", \"uniqueItems\": true, \"required\": true, \"items\": { \"$ref\": \"Playlist.Property.Name\" } }"
      "],"
      "\"returns\":  { \"$ref\": \"Playlist.Property.Value\", \"required\": true }"
    "}",
    "\"Playlist.GetItems\": {"
      "\"type\": \"method\","
      "\"description\": \"Get all items from playlist\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"playlistid\", \"$ref\": \"Playlist.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"List.Fields.All\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"items\": { \"type\": \"array\", \"items\": { \"$ref\": \"List.Item.All\" }, \"required\": true }"
        "}"
      "}"
    "}",
    "\"Playlist.Add\": {"
      "\"type\": \"method\","
      "\"description\": \"Add item(s) to playlist\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playlistid\", \"$ref\": \"Playlist.Id\", \"required\": true },"
        "{ \"name\": \"item\", \"$ref\": \"Playlist.Item\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Playlist.Insert\": {"
      "\"type\": \"method\","
      "\"description\": \"Insert item(s) into playlist. Does not work for picture playlists (aka slideshows).\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playlistid\", \"$ref\": \"Playlist.Id\", \"required\": true },"
        "{ \"name\": \"position\", \"$ref\": \"Playlist.Position\", \"required\": true },"
        "{ \"name\": \"item\", \"$ref\": \"Playlist.Item\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Playlist.Remove\": {"
      "\"type\": \"method\","
      "\"description\": \"Remove item from playlist. Does not work for picture playlists (aka slideshows).\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playlistid\", \"$ref\": \"Playlist.Id\", \"required\": true },"
        "{ \"name\": \"position\", \"$ref\": \"Playlist.Position\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Playlist.Clear\": {"
      "\"type\": \"method\","
      "\"description\": \"Clear playlist\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playlistid\", \"$ref\": \"Playlist.Id\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Playlist.Swap\": {"
      "\"type\": \"method\","
      "\"description\": \"Swap items in the playlist. Does not work for picture playlists (aka slideshows).\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"playlistid\", \"$ref\": \"Playlist.Id\", \"required\": true },"
        "{ \"name\": \"position1\", \"$ref\": \"Playlist.Position\", \"required\": true },"
        "{ \"name\": \"position2\", \"$ref\": \"Playlist.Position\", \"required\": true }"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"Files.GetSources\": {"
      "\"type\": \"method\","
      "\"description\": \"Get the sources of the media windows\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"media\", \"$ref\": \"Files.Media\", \"required\": true },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"sources\": { \"$ref\": \"List.Items.Sources\", \"required\": true }"
        "}"
      "}"
    "}",
    "\"Files.PrepareDownload\": {"
      "\"type\": \"method\","
      "\"description\": \"Provides a way to download a given file (e.g. providing an URL to the real file location)\","
      "\"transport\": [ \"Response\", \"FileDownloadRedirect\" ],"
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"path\", \"type\": \"string\", \"required\": true }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"protocol\": { \"type\": \"string\", \"enum\": [ \"http\" ], \"required\": true },"
          "\"details\": { \"type\": \"any\", \"required\": true, \"description\": \"Transport specific details on how/from where to download the given file\" },"
          "\"mode\": { \"type\": \"string\", \"enum\": [ \"redirect\", \"direct\" ], \"required\": true, \"description\": \"Direct mode allows using Files.Download whereas redirect mode requires the usage of a different protocol\" }"
        "}"
      "}"
    "}",
    "\"Files.Download\": {"
      "\"type\": \"method\","
      "\"description\": \"Downloads the given file\","
      "\"transport\": [ \"Response\", \"FileDownloadDirect\" ],"
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"path\", \"type\": \"string\", \"required\": true }"
      "],"
      "\"returns\": { \"type\": \"any\", \"required\": true }"
    "}",
    "\"Files.GetDirectory\": {"
      "\"type\": \"method\","
      "\"description\": \"Get the directories and files in the given directory\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"directory\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"media\", \"$ref\": \"Files.Media\", \"default\": \"files\" },"
        "{ \"name\": \"properties\", \"$ref\": \"List.Fields.All\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"files\": { \"type\": \"array\", \"items\": { \"$ref\": \"List.Item.File\" }, \"required\": true }"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetArtists\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all artists\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"albumartistsonly\", \"$ref\": \"Optional.Boolean\", \"description\": \"Whether or not to include artists only appearing in compilations. If the parameter is not passed or is passed as null the GUI setting will be used\" },"
        "{ \"name\": \"genreid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Artist\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"artists\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Audio.Details.Artist\" }"
          "}"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetArtistDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific artist\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"artistid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Artist\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"artistdetails\": { \"$ref\": \"Audio.Details.Artist\" }"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetAlbums\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all albums from specified artist or genre\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"artistid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"genreid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Album\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"albums\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Audio.Details.Album\" }"
          "}"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetAlbumDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific album\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"albumid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Album\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"albumdetails\": { \"$ref\": \"Audio.Details.Album\" }"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetSongs\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all songs from specified album, artist or genre\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"artistid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"albumid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"genreid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Song\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"songs\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Audio.Details.Song\" }"
          "}"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetSongDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific song\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"songid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Song\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"songdetails\": { \"$ref\": \"Audio.Details.Song\" }"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetRecentlyAddedAlbums\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve recently added albums\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Album\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"albums\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Audio.Details.Album\" }"
          "}"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetRecentlyAddedSongs\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve recently added songs\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"albumlimit\", \"$ref\": \"List.Amount\", \"description\": \"The amount of recently added albums from which to return the songs\" },"
        "{ \"name\": \"properties\", \"$ref\": \"Audio.Fields.Song\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"songs\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Audio.Details.Song\" }"
          "}"
        "}"
      "}"
    "}",
    "\"AudioLibrary.GetGenres\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all genres\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Library.Fields.Genre\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"genres\": { \"type\": \"array\", \"required\": true,"
            "\"items\": { \"$ref\": \"Library.Details.Genre\" }"
          "}"
        "}"
      "}"
    "}",
    "\"AudioLibrary.Scan\": {"
      "\"type\": \"method\","
      "\"description\": \"Scans the audio sources for new library items\","
      "\"transport\": \"Response\","
      "\"permission\": \"UpdateData\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"AudioLibrary.Export\": {"
      "\"type\": \"method\","
      "\"description\": \"Exports all items from the audio library\","
      "\"transport\": \"Response\","
      "\"permission\": \"WriteFile\","
      "\"params\": ["
        "{ \"name\": \"options\", \"type\": ["
            "{ \"type\": \"object\", \"required\": true, \"additionalProperties\": false,"
              "\"properties\": {"
                "\"path\": { \"type\": \"string\", \"required\": true, \"minLength\": 1, \"description\": \"Path to the directory to where the data should be exported\" }"
              "}"
            "},"
            "{ \"type\": \"object\", \"required\": true, \"additionalProperties\": false,"
              "\"properties\": {"
                "\"overwrite\": { \"type\": \"boolean\", \"default\": false, \"description\": \"Whether to overwrite existing exported files\" },"
                "\"images\": { \"type\": \"boolean\", \"default\": false, \"description\": \"Whether to export thumbnails and fanart images\" }"
              "}"
            "}"
          "]"
        "}"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"AudioLibrary.Clean\": {"
      "\"type\": \"method\","
      "\"description\": \"Cleans the audio library from non-existent items\","
      "\"transport\": \"Response\","
      "\"permission\": \"RemoveData\","
      "\"params\": [ ],"
      "\"returns\": \"string\""
    "}",
    "\"VideoLibrary.GetMovies\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all movies\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.Movie\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"movies\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.Movie\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetMovieDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific movie\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"movieid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.Movie\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"moviedetails\": { \"$ref\": \"Video.Details.Movie\" }"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetMovieSets\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all movie sets\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.MovieSet\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"sets\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.MovieSet\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetMovieSetDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific movie set\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"setid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.MovieSet\" },"
        "{ \"name\": \"movies\", \"type\": \"object\","
          "\"properties\": {"
            "\"properties\": { \"$ref\": \"Video.Fields.Movie\" },"
            "\"limits\": { \"$ref\": \"List.Limits\" },"
            "\"sort\": { \"$ref\": \"List.Sort\" }"
          "}"
        "}"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"setdetails\": { \"$ref\": \"Video.Details.MovieSet.Extended\" }"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetTVShows\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all tv shows\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.TVShow\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"tvshows\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.TVShow\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetTVShowDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific tv show\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"tvshowid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.TVShow\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"tvshowdetails\": { \"$ref\": \"Video.Details.TVShow\" }"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetSeasons\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all tv seasons\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"tvshowid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.Season\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"seasons\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.Season\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetEpisodes\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all tv show episodes\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"tvshowid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"season\", \"type\": \"integer\", \"minimum\": 0, \"default\": -1 },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.Episode\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"episodes\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.Episode\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetEpisodeDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific tv show episode\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"episodeid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.Episode\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"episodedetails\": { \"$ref\": \"Video.Details.Episode\" }"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetMusicVideos\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all music videos\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"artistid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"albumid\", \"$ref\": \"Library.Id\" },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.MusicVideo\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"musicvideos\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.MusicVideo\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetMusicVideoDetails\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve details about a specific music video\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"musicvideoid\", \"$ref\": \"Library.Id\", \"required\": true },"
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.MusicVideo\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"musicvideodetails\": { \"$ref\": \"Video.Details.MusicVideo\" }"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetRecentlyAddedMovies\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all recently added movies\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.Movie\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"movies\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.Movie\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetRecentlyAddedEpisodes\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all recently added tv episodes\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.Episode\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"episodes\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.Episode\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetRecentlyAddedMusicVideos\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all recently added music videos\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"$ref\": \"Video.Fields.MusicVideo\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": { \"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"musicvideos\": { \"type\": \"array\","
            "\"items\": { \"$ref\": \"Video.Details.MusicVideo\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.GetGenres\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve all genres\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"type\", \"type\": \"string\", \"required\": true, \"enum\": [ \"movie\", \"tvshow\", \"musicvideo\"] },"
        "{ \"name\": \"properties\", \"$ref\": \"Library.Fields.Genre\" },"
        "{ \"name\": \"limits\", \"$ref\": \"List.Limits\" },"
        "{ \"name\": \"sort\", \"$ref\": \"List.Sort\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"properties\": {"
          "\"limits\": { \"$ref\": \"List.LimitsReturned\", \"required\": true },"
          "\"genres\": { \"type\": \"array\", \"required\": true,"
            "\"items\": { \"$ref\": \"Library.Details.Genre\" }"
          "}"
        "}"
      "}"
    "}",
    "\"VideoLibrary.Scan\": {"
      "\"type\": \"method\","
      "\"description\": \"Scans the video sources for new library items\","
      "\"transport\": \"Response\","
      "\"permission\": \"UpdateData\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"VideoLibrary.Export\": {"
      "\"type\": \"method\","
      "\"description\": \"Exports all items from the video library\","
      "\"transport\": \"Response\","
      "\"permission\": \"WriteFile\","
      "\"params\": ["
        "{ \"name\": \"options\", \"type\": ["
            "{ \"type\": \"object\", \"required\": true, \"additionalProperties\": false,"
              "\"properties\": {"
                "\"path\": { \"type\": \"string\", \"required\": true, \"minLength\": 1, \"description\": \"Path to the directory to where the data should be exported\" }"
              "}"
            "},"
            "{ \"type\": \"object\", \"required\": true, \"additionalProperties\": false,"
              "\"properties\": {"
                "\"overwrite\": { \"type\": \"boolean\", \"default\": false, \"description\": \"Whether to overwrite existing exported files\" },"
                "\"images\": { \"type\": \"boolean\", \"default\": false, \"description\": \"Whether to export thumbnails and fanart images\" },"
                "\"actorthumbs\": { \"type\": \"boolean\", \"default\": false, \"description\": \"Whether to export actor thumbnails\" }"
              "}"
            "}"
          "]"
        "}"
      "],"
      "\"returns\": \"string\""
    "}",
    "\"VideoLibrary.Clean\": {"
      "\"type\": \"method\","
      "\"description\": \"Cleans the video library from non-existent items\","
      "\"transport\": \"Response\","
      "\"permission\": \"RemoveData\","
      "\"params\": [ ],"
      "\"returns\": \"string\""
    "}",
    "\"System.GetProperties\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieves the values of the given properties\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"type\": \"array\", \"uniqueItems\": true, \"required\": true, \"items\": { \"$ref\": \"System.Property.Name\" } }"
      "],"
      "\"returns\":  { \"$ref\": \"System.Property.Value\", \"required\": true }"
    "}",
    "\"System.Shutdown\": {"
      "\"type\": \"method\","
      "\"description\": \"Shuts the system running XBMC down\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPower\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"System.Suspend\": {"
      "\"type\": \"method\","
      "\"description\": \"Suspends the system running XBMC\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPower\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"System.Hibernate\": {"
      "\"type\": \"method\","
      "\"description\": \"Puts the system running XBMC into hibernate mode\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPower\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"System.Reboot\": {"
      "\"type\": \"method\","
      "\"description\": \"Reboots the system running XBMC\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPower\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Input.Left\": {"
      "\"type\": \"method\","
      "\"description\": \"Navigate left in GUI\","
      "\"transport\": \"Response\","
      "\"permission\": \"Navigate\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Input.Right\": {"
      "\"type\": \"method\","
      "\"description\": \"Navigate right in GUI\","
      "\"transport\": \"Response\","
      "\"permission\": \"Navigate\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Input.Down\": {"
      "\"type\": \"method\","
      "\"description\": \"Navigate down in GUI\","
      "\"transport\": \"Response\","
      "\"permission\": \"Navigate\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Input.Up\": {"
      "\"type\": \"method\","
      "\"description\": \"Navigate up in GUI\","
      "\"transport\": \"Response\","
      "\"permission\": \"Navigate\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Input.Select\": {"
      "\"type\": \"method\","
      "\"description\": \"Select current item in GUI\","
      "\"transport\": \"Response\","
      "\"permission\": \"Navigate\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Input.Back\": {"
      "\"type\": \"method\","
      "\"description\": \"Goes back in GUI\","
      "\"transport\": \"Response\","
      "\"permission\": \"Navigate\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Input.Home\": {"
      "\"type\": \"method\","
      "\"description\": \"Goes to home window in GUI\","
      "\"transport\": \"Response\","
      "\"permission\": \"Navigate\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"Application.GetProperties\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieves the values of the given properties\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"properties\", \"type\": \"array\", \"uniqueItems\": true, \"required\": true, \"items\": { \"$ref\": \"Application.Property.Name\" } }"
      "],"
      "\"returns\":  { \"$ref\": \"Application.Property.Value\", \"required\": true }"
    "}",
    "\"Application.SetVolume\": {"
      "\"type\": \"method\","
      "\"description\": \"Set the current volume\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"volume\", \"type\": \"integer\", \"minimum\": 0, \"maximum\": 100, \"required\": true }"
      "],"
      "\"returns\": \"integer\""
    "}",
    "\"Application.SetMute\": {"
      "\"type\": \"method\","
      "\"description\": \"Toggle mute/unmute\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPlayback\","
      "\"params\": ["
        "{ \"name\": \"mute\", \"required\": true, \"type\": ["
            "{ \"type\": \"boolean\", \"required\": true },"
            "{ \"type\": \"string\", \"enum\": [ \"toggle\" ], \"required\": true }"
          "]"
        "}"
      "],"
      "\"returns\": { \"type\": \"boolean\", \"description\": \"Mute state\" }"
    "}",
    "\"Application.Quit\": {"
      "\"type\": \"method\","
      "\"description\": \"Quit application\","
      "\"transport\": \"Response\","
      "\"permission\": \"ControlPower\","
      "\"params\": [],"
      "\"returns\": \"string\""
    "}",
    "\"XBMC.GetInfoLabels\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve info labels about XBMC and the system\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"labels\", \"type\": \"array\", \"required\": true, \"items\": { \"type\": \"string\" }, \"minItems\": 1, \"description\": \"See http://wiki.xbmc.org/index.php?title=InfoLabels for a list of possible info labels\" }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"description\": \"List of key-value pairs of the retrieved info labels\""
      "}"
    "}",
    "\"XBMC.GetInfoBooleans\": {"
      "\"type\": \"method\","
      "\"description\": \"Retrieve info booleans about XBMC and the system\","
      "\"transport\": \"Response\","
      "\"permission\": \"ReadData\","
      "\"params\": ["
        "{ \"name\": \"booleans\", \"type\": \"array\", \"required\": true, \"items\": { \"type\": \"string\" }, \"minItems\": 1 }"
      "],"
      "\"returns\": {"
        "\"type\": \"object\","
        "\"description\": \"List of key-value pairs of the retrieved info booleans\""
      "}"
    "}"
  };

  const char* const JSONRPC_SERVICE_NOTIFICATIONS[] = {  
    "\"Player.OnPlay\": {"
      "\"type\": \"notification\","
      "\"description\": \"Playback of a media item has been started or the playback speed has changed. If there is no ID available extra information will be provided.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"$ref\": \"Player.Notifications.Data\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}",
    "\"Player.OnPause\": {"
      "\"type\": \"notification\","
      "\"description\": \"Playback of a media item has been paused. If there is no ID available extra information will be provided.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"$ref\": \"Player.Notifications.Data\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}",
    "\"Player.OnStop\": {"
      "\"type\": \"notification\","
      "\"description\": \"Playback of a media item has been stopped. If there is no ID available extra information will be provided.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"object\", \"required\": true, "
          "\"properties\": {"
            "\"item\": { \"$ref\": \"Player.Notifications.Item\" }"
          "}"
        "}"
      "],"
      "\"returns\": null"
    "}",
    "\"Player.OnSpeedChanged\": {"
      "\"type\": \"notification\","
      "\"description\": \"Speed of the playback of a media item has been changed. If there is no ID available extra information will be provided.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"$ref\": \"Player.Notifications.Data\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}",
    "\"Player.OnSeek\": {"
      "\"type\": \"notification\","
      "\"description\": \"The playback position has been changed. If there is no ID available extra information will be provided.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"object\", \"required\": true, "
          "\"properties\": {"
            "\"item\": { \"$ref\": \"Player.Notifications.Item\" },"
            "\"player\": { \"$ref\": \"Player.Notifications.Player.Seek\", \"required\": true }"
          "}"
        "}"
      "],"
      "\"returns\": null"
    "}",
    "\"AudioLibrary.OnUpdate\": {"
      "\"type\": \"notification\","
      "\"description\": \"An audio item has been updated.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"object\", \"required\": true,"
          "\"properties\": {"
            "\"id\": { \"$ref\": \"Library.Id\", \"required\": true },"
            "\"type\": { \"type\": \"string\", \"id\": \"Notifications.Library.Audio.Type\", \"enum\": [ \"song\" ], \"required\": true }"
          "}"
        "}"
      "],"
      "\"returns\": null"
    "}",
    "\"AudioLibrary.OnRemove\": {"
      "\"type\": \"notification\","
      "\"description\": \"An audio item has been removed.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"object\", \"required\": true,"
          "\"properties\": {"
            "\"id\": { \"$ref\": \"Library.Id\", \"required\": true },"
            "\"type\": { \"$ref\": \"Notifications.Library.Audio.Type\", \"required\": true }"
          "}"
        "}"
      "],"
      "\"returns\": null"
    "}",
    "\"VideoLibrary.OnUpdate\": {"
      "\"type\": \"notification\","
      "\"description\": \"A video item has been updated.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"object\", \"required\": true,"
          "\"properties\": {"
            "\"id\": { \"$ref\": \"Library.Id\", \"required\": true },"
            "\"type\": { \"type\": \"string\", \"id\": \"Notifications.Library.Video.Type\", \"enum\": [ \"movie\", \"tvshow\", \"episode\", \"musicvideo\" ], \"required\": true },"
            "\"playcount\": { \"type\": \"integer\", \"minimum\": 0, \"default\": -1 }"
          "}"
        "}"
      "],"
      "\"returns\": null"
    "}",
    "\"VideoLibrary.OnRemove\": {"
      "\"type\": \"notification\","
      "\"description\": \"A video item has been removed.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"object\", \"required\": true,"
          "\"properties\": {"
            "\"id\": { \"$ref\": \"Library.Id\", \"required\": true },"
            "\"type\": { \"$ref\": \"Notifications.Library.Video.Type\", \"required\": true }"
          "}"
        "}"
      "],"
      "\"returns\": null"
    "}",
    "\"System.OnQuit\": {"
      "\"type\": \"notification\","
      "\"description\": \"XBMC will be closed.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"null\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}",
    "\"System.OnRestart\": {"
      "\"type\": \"notification\","
      "\"description\": \"The system will be restarted.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"null\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}",
    "\"System.OnSleep\": {"
      "\"type\": \"notification\","
      "\"description\": \"The system will be suspended.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"null\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}",
    "\"System.OnWake\": {"
      "\"type\": \"notification\","
      "\"description\": \"The system woke up from suspension.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"null\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}",
    "\"System.OnLowBattery\": {"
      "\"type\": \"notification\","
      "\"description\": \"The system is on low battery.\","
      "\"params\": ["
        "{ \"name\": \"sender\", \"type\": \"string\", \"required\": true },"
        "{ \"name\": \"data\", \"type\": \"null\", \"required\": true }"
      "],"
      "\"returns\": null"
    "}"
  };
}
