#!/usr/bin/env bash


# Initialize variables
cpuUsage=0
cpuTemp=0
memoryUsed=0
memoryTotal=0
memoryPercent=0
swapUsed=0
swapTotal=0
swapPercent=0
diskUsed=0
diskTotal=0
diskPercent=0
networkSent=0
networkRecv=0

# Helper function for temperature conversion
celsius_to_fahrenheit() {
    local celsius="$1"
    [[ -z "$celsius" || "$celsius" == "--" ]] && echo "--" && return
    [[ ! "$celsius" =~ ^-?[0-9]+(\.[0-9]+)?$ ]] && echo "--" && return
    
    local fahrenheit
    fahrenheit=$(echo "scale=1; ($celsius * 9/5) + 32" | bc -l)
    echo "$fahrenheit"
}

update_system_data() {
    # CPU usage
    cpuUsage=$(top -bn1 | grep "Cpu(s)" | sed "s/.*, *\([0-9.]*\)%* id.*/\1/" | awk '{print 100 - $1}')
    cpuUsage=$(printf "%.1f" "$cpuUsage" 2>/dev/null || echo "$cpuUsage")
    
    # CPU temperature - fix for high readings
    local tempCelsius="--"
    if [[ -f /sys/class/thermal/thermal_zone0/temp ]]; then
        tempCelsius=$(cat /sys/class/thermal/thermal_zone0/temp)
        # Check if we need to divide by 1000 (common format)
        if [[ $tempCelsius -gt 200 ]]; then
            tempCelsius=$(echo "scale=1; $tempCelsius/1000" | bc -l)
        fi
    elif [[ -f /sys/class/hwmon/hwmon0/temp1_input ]]; then
        tempCelsius=$(cat /sys/class/hwmon/hwmon0/temp1_input)
        if [[ $tempCelsius -gt 200 ]]; then
            tempCelsius=$(echo "scale=1; $tempCelsius/1000" | bc -l)
        fi
    elif command -v sensors &>/dev/null; then
        tempCelsius=$(sensors | grep -oP 'Core 0.*?\+\K[0-9.]+')
    fi
    
    [[ "$tempCelsius" != "--" && "$tempCelsius" != "null" ]] && cpuTemp=$(celsius_to_fahrenheit "$tempCelsius") || cpuTemp="--"
    
    # Memory usage
    memoryTotal=$(free -m | awk '/^Mem:/{print $2/1024}')
    memoryUsed=$(free -m | awk '/^Mem:/{print $3/1024}')
    memoryPercent=$(free | awk '/^Mem:/{printf("%.1f", $3/$2 * 100)}')
    
    # Swap usage
    swapTotal=$(free -m | awk '/^Swap:/{print $2/1024}')
    swapUsed=$(free -m | awk '/^Swap:/{print $3/1024}')
    [[ "$swapTotal" -gt 0 ]] && swapPercent=$(free | awk '/^Swap:/{printf("%.1f", $3/$2 * 100)}') || swapPercent=0
    
    # Disk usage
    diskTotal=$(df -h / | awk 'NR==2 {print $2}')
    diskUsed=$(df -h / | awk 'NR==2 {print $3}')
    diskPercent=$(df -h / | awk 'NR==2 {print $5}' | tr -d '%')
    
    # Network usage
    networkSent=$(cat /proc/net/dev | grep -v "lo:" | awk '{s+=$10} END {printf "%.2f", s/1024/1024}')
    networkRecv=$(cat /proc/net/dev | grep -v "lo:" | awk '{r+=$2} END {printf "%.2f", r/1024/1024}')
}

display_system_info() {
    update_system_data
    
    # Create the system information text with proper line breaks
    systemInfoText="System Information

CPU
Usage: ${cpuUsage}%
Temperature: ${cpuTemp}° F

Memory
Usage: ${memoryUsed} / ${memoryTotal} GB (${memoryPercent}%)
Swap: ${swapUsed} / ${swapTotal} GB (${swapPercent}%)

Disk (Root Partition)
Usage: ${diskUsed} / ${diskTotal} GB (${diskPercent}%)

Network (Total Since Boot)
Received: ${networkRecv} MB
Sent: ${networkSent} MB

End of text. Press Control+Home to return to the beginning."

    # Display in text-info dialog for screen reader accessibility
    echo "$systemInfoText" | yad --pname=I38System \
        --title="I38 System Information" \
        --text-info \
        --show-cursor \
        --width=400 \
        --height=400 \
        --center \
        --button="Close:0"
}

display_system_info

exit 0
