#!/usr/bin/env bash

# This file is part of I38.
# 
# I38 is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation,
# either version 3 of the License, or (at your option) any later version.
#                                                                                                                                                                          
# I38 is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#                                                                                                                                                                          
# You should have received a copy of the GNU General Public License along with I38. If not, see <https://www.gnu.org/licenses/>.

is_running() {
    pgrep -x "$1" >/dev/null
    return $?
}

speak() {
    spd-say -P important -Cw -- "$*"
}

# Make sure screen readers are available
orcaAvailable=false
cthulhuAvailable=false

if command -v orca &> /dev/null; then
    orcaAvailable=true
fi

if command -v cthulhu &> /dev/null; then
    cthulhuAvailable=true
fi

# Check if at least one screen reader is available
if ! $orcaAvailable && ! $cthulhuAvailable; then
    speak "No screen readers found. Please install Orca or Cthulhu."
    yad --center --title="I38" --alert="No screen readers found.\nPlease install Orca or Cthulhu."
    exit 1
fi

# Determine current state
currentReader="none"
if is_running "cthulhu"; then
    currentReader="cthulhu"
elif is_running "orca"; then
    currentReader="orca"
fi

# Build YAD command based on available screen readers
items=()

# First add the currently active screen reader
if [ "$currentReader" != "none" ]; then
    if [ "$currentReader" = "cthulhu" ] && $cthulhuAvailable; then
        items+=("Cthulhu (active)" "cthulhu")
    elif [ "$currentReader" = "orca" ] && $orcaAvailable; then
        items+=("Orca (active)" "orca")
    fi
fi

if $cthulhuAvailable && [ "$currentReader" != "cthulhu" ]; then
    items+=("Cthulhu" "cthulhu")
fi

if $orcaAvailable && [ "$currentReader" != "orca" ]; then
    items+=("Orca" "orca")
fi

# Display the dialog
result=$(yad --center --title="Screen Reader Toggle" --text="Select screen reader:" \
    --list --on-top --skip-taskbar \
    --button="Cancel:1" --button="OK:0" \
    --column="Screen Reader" --column=ID:HD "${items[@]}")
exitCode=$?
if [ $exitCode -ne 0 ]; then
    exit 0
fi

if [ -n "$result" ]; then
    # Extract the selected reader from the result
    selectedReader=$(echo "$result" | cut -d'|' -f2)
    
    # Don't do anything if selecting the already active reader
    if [ "$selectedReader" = "$currentReader" ]; then
        exit 0
    fi
    
    # Stop current screen reader
    if [ "$currentReader" != "none" ]; then
        speak "Switching from $currentReader to $selectedReader."
        pkill -15 "$currentReader"
        sleep 0.5
    else
        speak "Starting $selectedReader."
    fi
    
    if [ "$selectedReader" = "orca" ]; then
        orca &
    else
        cthulhu &
    fi
fi

exit 0
