#!/usr/bin/wish -f

set tki_header {

tkInfo version 0.6				VERSION
Author: Kennard White <kennard@ohm.berkeley.edu>
RCS: $Header: /vol/cooley/cooley1/kennard/src/tkgraph/lib/RCS/tkinfo.tcl,v 1.14 93/08/27 00:25:50 kennard Exp $

Tk script to read GNU "info" files and display them.  tkInfo can be
used stand alone (via WISH), or embedded within an application to provide
integrated, on-line help.

Please see the "About" and "Info" file sections below.  (search for
"README" to find these sections quickly).  These explain much more
about what tkinfo is and what info files are, and gives references to
other programs and sources of info.

tkinfo consists of the files:
	tkinfo.tcl (the core file, and the one you are reading now)
	topgetopt.tcl (argument parsing)
	searchbox.tcl (text searching)
	tkinfo	(shell script for invoking tkinfo.tcl with wish)
All should be in the same directory.
This release works with tcl6.6/tk3.1, tcl6.7/tk3.2 and tcl7.0b2/tk3.3b2.
tkinfo has gone through several releases, but it is by no means complete.
Feel free to make suggestions, or better yet, send me patch files.

See below for copyright (search for "README").  Basically you can
re-distribute this any way you like, just dont sue me and dont pretend
you wrote tkInfo.  Portions of the GNU "makeinfo" is GNU copylefted,
but only the author of the document needs "makeinfo": the author can
freely redistribute the info files produced by "makeinfo" (the ones
read by this script).

Contributions and/or good ideas by Larry Virden <lvirden@cas.org>,
Bob Bagwill <bagwill@swe.ncsl.nist.gov>, ??? <tlukka@snakemail.hut.fi>,
Kurt Hornik <hornik@neuro.tuwien.ac.at>.
Tom Phelps <phelps@CS.Berkeley.EDU> contributed the searching code,
as well as other good ideas.

} ; set tki_header {

		Stand-alone usage, via WISH

When invoked with no arguments, tkinfo looks for an "info tree"
(a collection of info files installed on your system) and displays
the top level node.  On a well maintained system, you can get
to every info file starting from this top level node.  Alternatively,
you can specify the file you want to see on the command line as:
"tkinfo -infofile <filename>".  The complete syntax is below:

usage: tkinfo [-headers] [-nodelook looktype] [-dir dir1] [-dir dir2] ... \
		  [-infofile filePath] [node]
Options:
   -headers	Turns on display of the raw info node headers.
   -nodelook	Specifies how to highlight xrefs and menu entries.  Must
		be one of "color", "font", or "underline".
   -dir	Specifies directories to search for info files, like the
		INFOPATH variable.
   -infofile	Specifies a file to start with.
   node	Specifies the node you want to see.  The complete syntax
		is below, but it will generally be of the form
		"(filename)nodename"; remember to use quotes if nodename
		has spaces in it.
Note that the "-file" argument is taken by WISH, and thus cant be used
to specify the info file name.
Environment variables:
   INFOPATH	A colon (``:'') seperated list of directories to search
		for info files.  Equivalent to each -dir option above.
   INFOSUFFIX	A colon seperated list of file suffixes to try when searching
		for an info file.  tkinfo will automatically try the
		suffixes .z and .Z and deal with the compressed files
		if required.
The default INFOPATH and INFOSUFFIX is defined in tkiInit() below.

} ; set tki_header {

Embedded usage:
Set the global variable ``tkiEmbed''.  Source this file.
If your info files are installed in a non-standard place,
call proc tkiAddInfoPaths.  It takes one argument, which should
be a directory to search for info files.
To popup an info window, call tkiGetAndDisplay,
passing it a nodeSpec (see below) of what to display.
The nodeSpec may be an empty string, in which case the highest level
node in the system is displayed.

This script requires "topgetopt.tcl", which should be installed in
the same directory as this script.  In stand-alone mode, this file is
directly source'd. In embedded mode, tkinfo depends on the auto-load
feature (so be sure to update the tclIndex).

This file is organized in 3 parts:
  1)  This comment area, which briefly describes the info file format
	and this implementation.
  2)	Code for parsing info files.  This code is independent of tk.
  3)	Code for displaying info files/nodes.  This calls the parsing
	functions and then displays the results in various tk widgets.

The program provides on-line help of itself: node (builtin)QuickHelp,
and the texinfo distribution contains a tutorial: node (info)Top.

} ; unset tki_header

# tkInfo requires the following global variables:
#  tki		This is a huge array where essentially all the loaded info-files
#		are stored.  It also contains some configuration state.
#		The contents of this is described below.
#  .tki##	Each toplevel info window has a global variable associated
#		with it.  The name of the variable is the same as the
#		toplevel window name, which is ".tki" followed by some number.
#  tkiEmbed	tkInfo can operate stand-alone (like the "info" program) or
#		embedded (part of your application).  Embedded mode is
#		used iff this variable exists.  When this file is
#		sourced in the stand-alone case, the argv options will be
#		parsed (see tkiBoot() below) and a new toplevel window
#	 	will be opened.

# The core structure of an info file is a {node}.  Each info file consists
# of a set of nodes seperated by a magic character.  Each nodes consists of
# of a header, body, and menu.  There are also special nodes that contain
# control information used to reference "split" files and speed up access.
# The first half of the code below parses this file format.
# A node may be specified in one of several ways (called a {nodeSpec}):
#	(filename)nodename
#	nodename		The given node within the current file.
#	(filename)		The "Top" node of the file.
#	(filename)*		The entire file.


# In the implementation below, the info format consists of {nodes} stored 
# in files.  A given info file has three identifiers associated with it:
#  -	The {filename}, which is the name used either by the user to
#	reference a file, or by one info file to reference another.
#	Such a reference could be complete UNIX path name (either
#	absolute or relative), or may be a partial specification (see below).
#  -	The {filepath}, which is a valid UNIX path name to access the
#	file.  The filepath is derived from the filename.  If the filename
#	is already a valid path, no work needs be done.  Otherwise,
#	the filepath is formed by prepending a path prefix and appending
#	a file suffix.  These are defined by the INFOPATH and INFOSUFFIX
#	variables.
#  -	The {filekey}, which is an internal, auto-generated token associated
#	with each file.


#
# The global array "tki" contains the following elements:
#   file-$fileName	The fileKey for $fileName.
#   fileinfo-$fileKey	The fileinfo struct for $fileKey. Each fileinfo is
#			{ fileKey fileName filePath pntKey wholeB }
#   incore-$fileKey	Boolean 0/1; true if file has been loaded into core.
#   nodesinfo-$fileKey	A list of nodeinfo for every node in $fileKey.
#			Each nodeinfo is a list { idx node file up prev next }.
#   nodesbody-$fileKey	A list of the textual body for every node in $fileKey.
#   indirf-$fileKey	List of indirect-file-info for $fileKey.  Each
#			info is a list { indirFileKey byteOfs }.
#   indirn-$fileKey	List of indirect-node-info for $fileKey.  Each
#			info is a list { nodeName byteOfs fileKey }.
#   xrefinfo-$fileKey-$nodeIdx
#			Contains information on all cross reference
#			pointers within the node's body text.
#   menuinfo-$fileKey-$nodeIdx
#			Contains information on all menu entries
#			within the node's menu text.  Constist of list of:
#			 { linecnt menucnt toNode nBeg nEnd }
#
#
# Notes (some important, some not).
# 1.	Because of the graphical system, there may be several parallel
#	info windows active.  These windows must operate independently.
#	Because of this, there can be no concept of the "current file"
#	or "current node" within the tkinfo core.  Rather, this information
#	must be maintained by the window.
# 2.	Because of #1, we must maintain multiple files in core.  Currently
#	we never flush.
# 3.	The background color used in tkiInit() is BISQUE1, from tk/defaults.h
# 4.	The byte offsets in the indirect tables are not used as such;
#	this is because we parse the file when loaded.  However, they are
#	used to identify which indirect file the node is in.
# 5.	The function tkiLoadFile() attempts to deal with compressed files.
#	Currently it uses "zcat" for .Z files and "gunzip -c" for .z files.
#	If you have better suggestions, please let me know.
#

# Ignore this, it is used by a custom auto-reload script.
proc tkinfo.tcl { } { }

proc tkiInit { } {
    global tki env auto_path tkiEmbed

    set defInfoPath [list . /usr/tools/gnu/info /usr/gnu/info /usr/local/emacs/info /usr/local/lib/gnumacs/info /usr/local/lib/emacs/info]
#    set defInfoPath [list . ]
    set defInfoSuffix [list "" .info -info]

    if { [info exist tki] } return

    set tki(self)		[info script]
    set tki(sn)		0
    set tki(compresscat-Z)	"zcat"
    set tki(compresscat-z)	"gunzip -c"
    set tki(background)		"#ffe4c4"
    set tki(rawHeadersB)	0
    set tki(nodeSep)		"\037"
    set tki(nodeByteSep)	"\177"
    set tki(topLevelFile)	"dir"
    set tki(topLevelNode)	"Top"

    set tki(nodelook)		"color"
    if { "[info commands winfo]"!="" } {
	if { [winfo depth .] == 1 } {
	    set tki(nodelook) underline
	}
    }
    set tki(nodelookColor)	"blue"
    set tki(nodelookFont)	"-*-Courier-Bold-O-Normal-*-*-120-*-*-*-*-*-*"
    set tki(normCursor)		"left_ptr"
    set tki(waitCursor)		"watch"
    set tki(windows)		""
    set tki(curWindow)		""
    if [info exist env(INFOPATH)] {
      tkiAddInfoPaths [split $env(INFOPATH) ":"]
    } else {
      tkiAddInfoPaths $defInfoPath
    }
    if [info exist env(INFOSUFFIX)] {
	set tki(infoSuffix) [split $env(INFOSUFFIX) ":"]
    } else {
	set tki(infoSuffix) $defInfoSuffix
    }
    _tkiSourceFile topgetopt.tcl
#    _tkiSourceFile taputils.tcl TextSearch
    _tkiSourceFile searchbox.tcl searchboxRegexpSearch

    _tkiNodeParseInit
    rename _tkiNodeParseInit ""

    _tkiBuiltinFile
    rename _tkiBuiltinFile ""

    trace var tki(rawHeadersB) w "_tkiTraceOptionsCB"
    trace var tki(nodelook) w "_tkiTraceOptionsCB"
}

proc _tkiTraceOptionsCB { n1 n2 op } {
    global tki
    foreach w $tki(windows) {
	if { ![winfo exist $w] } continue
#puts stdout "_tkiTraceOptionsCB: $w"
	if [catch {tkiReDpyWin $w} error] {
	    global errorInfo
	    puts stderr "tkInfo: $error\n$errorInfo"
	}
    }
}

#
# Add list tcl list of paths {newPaths} to the directory search list.
# The list is added in order at the *head* of the list.
# Duplicate paths are removed, leaving the first most path present.
#
proc tkiAddInfoPaths { newPaths } {
    global tki

    if { ! [info exist tki(infoPath) ] } {
	set tki(infoPath) ""
    }
    if { [llength $newPaths] > 0 } {
	set tki(infoPath) [eval linsert {$tki(infoPath)} 0 $newPaths]
    }
    # Kill off null paths
    while 1 {
	set idx [lsearch $tki(infoPath) ""]
	if { $idx < 0 }	break
	set tki(infoPath) [lreplace $tki(infoPath) $idx $idx]
    }
    # Kill off duplicate paths
    for {set idx 0} {$idx < [llength $tki(infoPath)]} {incr idx} {
	set path [lindex $tki(infoPath) $idx]
	while 1 {
	    set dup [lsearch [lrange $tki(infoPath) [expr {$idx+1} ] end] $path]
	    if { $dup < 0 } break
	    set tki(infoPath) [lreplace $tki(infoPath) $dup $dup]
	}
    }
}


# We need to be able to find varous files.  Assume its in
# the same directory as us.  Could just do:
#	lappend auto_path [file dirname $tki(self)]
# but this requires the tclIndex to be up-to-date, which is to
# much to ask of some users.  So just source it here.
proc _tkiSourceFile { filename {tagproc ""} } {
    global tki auto_path
    if { "$tagproc" == "" } { set tagproc $filename }
    if { "[info procs $tagproc]"=="" } {
	set selfdir [file dirname $tki(self)]
	set otherfile $selfdir/$filename
	if { [file isfile $otherfile] } {
	    if { [info exist tkiEmbed] } {
		if { [lsearch $auto_path $selfdir]==-1 } {
		    lappend auto_path $selfdir
		}
	    } else {
		uplevel #0 source $otherfile
	    }
	}
    }
}

proc tkiGetSN { } {
    global tki

    incr tki(sn)
    return $tki(sn)
}

proc tkiUninit { } {
    global tki
    catch {unset tki}
}

proc tkiReset { } {
    tkiUninit
    tkiInit
}

proc _tkiNull { args } {
}

# These are called from searchbox.tcl
proc winstdout {w msg} { tkiStatus $msg }
proc winerrout {w msg} { tkiError $msg }

proc tkiStatus { msg } {
    global tki
    if { "$tki(curWindow)"=="" } {
        puts stdout "tkInfo: $msg"
    } else {
	$tki(curWindow).s.status conf -text "$msg"
	# idletasks should be sufficient, but the geometry management
	# apparently needs some X-events to make the redisplay occur
	update
    }
}

proc tkiWarning { msg } {
    # Warnings allways go to stderr
    puts stderr "tkInfo Warning: $msg"
}

proc tkiFileWarning { fileSpec msg } {
    global tki
    if [info exist tki(fileinfo-$fileSpec)] {
	set fileSpec [lindex $tki(fileinfo-$fileSpec) 2]
    }
    tkiWarning "$fileSpec: $msg"
}

proc tkiError { msg } {
    global tki
    if [info exist tki(error-active)] {
	puts stderr "tkInfo Warning: Re-entrant error message!"
    }
    set tki(error-active) 1
    if { "$tki(curWindow)"=="" } {
        puts stdout "tkInfo Error: $msg"
    } else {
	set infowin $tki(curWindow)
	upvar #0 $infowin wvars
	set w .tkierr[tkiGetSN]
	if ![winfo ismapped $infowin] {
#puts stdout "tkiError: vis $infowin"
	    tkwait vis $infowin
	}
#puts stdout "tkiError: $w"
	toplevel $w
	wm geom $w +[winfo rootx $infowin]+[winfo rooty $infowin]
	wm title $w "tkInfo Error"
	wm iconname $w "tkInfo Error"
	pack append $w [label $w.title -text "tkInfo Error"] \
	  {top fillx}
	pack append $w [message $w.msg -text $msg \
	  -width [winfo width $infowin]] {top fill expand}
	pack append $w [button $w.dismiss -text "Dismiss" \
	  -com "tkiErrorDone $w $infowin"] {top fillx}
        $infowin.s.status conf -text "Error!"
	grab $w
#	$tki(curWindow).s.status conf -text "Error: $msg"
	tkwait window $w
        $infowin.s.status conf -text $wvars(status)
    }
    unset tki(error-active)
}

proc tkiErrorDone { errwin infowin } {
    catch {destroy $errwin}
}

#
# This proc is called once during initialization, and then destroyed.
# (It is destroyed to save memory).
# Currently we fake all the appropriate table entires to create a "builtin"
# file.  It might be easier, however, to just pass one large text string
# into the parser and have it be dealt with like any other file.
#
proc _tkiBuiltinFile { } {
    global tki

    set fileKey			builtin
    set tki(file-$fileKey)	$fileKey
    set tki(fileinfo-$fileKey)	[list $fileKey $fileKey $fileKey "" 0]
    set tki(incore-$fileKey)	1
    set tki(nodesinfo-$fileKey) ""
    set tki(nodesbody-$fileKey) ""

    set tki(dirnodes) ""
    set diritems "\nDirectory nodes (from INFOPATH):\n"
    foreach pp $tki(infoPath) {
	set dirpath [tkiFileFind $pp/$tki(topLevelFile)]
	if { "$dirpath" != "" } {
	    lappend tki(dirnodes) $dirpath
	    append diritems "* ($dirpath)::\n"
	}
    }
    if { [llength $tki(dirnodes)] == 0 } {
	append diritems "(No directories found!)\n"
    }

    tkiFileParseNode $fileKey "
File: builtin, Node: Top, Up: (dir)Top
tkInfo
------
This is the builtin info on tkInfo itself.

* Menu:
* Copyright::			tkInfo's copyright.
* About::			About tkInfo.
* Info::			Info files.
* QuickHelp::			Quick summary of how to use tkInfo.
* ToDo::			My to-do list.
* Source Code: ($tki(self))*	Source code to tkInfo.
$diritems"

#README
    tkiFileParseNode $fileKey {
File: builtin, Node: Copyright, Up: Top, Next: About
Copyright for tkInfo
--------------------
This copyright applies to the tkInfo system only.  If tkInfo is
embedded within a larger system, that system will most likely have
a different copyright.

Sorry this is so long.  Basically, do whatever you want with this
software, just don't sue me and don't pretend you wrote it -- kennard.

Copyright (c) 1993 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.
    }
#VERSION README
    tkiFileParseNode $fileKey {
File: builtin, Node: About, Up: Top, Prev: Copyright, Next: Info
About tkInfo-0.6
----------------
tkInfo version 0.6 (based on tcl-7.0 and tk-3.3) by Kennard White

tkInfo displays "info" files.  "Info" files provide a hyper-text
capability that is ideal for on-line help.  The format is suitable for
both tty-based systems and graphical systems.  In addition, the same
document source can produce both a "nice" hardcopy manual and Info
files.  For more information, *note (builtin)Info::.


tkInfo is the work of Kennard White.  For information, comments,
code-contributions or bug reports please contact me at:
	phone:	510/643-6686
	e-mail: kennard@ohm.eecs.Berkeley.EDU
    	s-mail:	Kennard White, Box #115
		207 Cory Hall
		UC Berkeley
		Berkeley, CA 94720, USA

You can obtain the tkinfo distribution by anonymous ftp from:
	ftp@ptolemy.eecs.berkeley.edu:/pub/misc
	ftp@harbor.ecn.purdue.edu:/pub/tcl/code
The first site will always have the most recent version.

RCS: $Revision: 1.14 $, $Date: 93/08/27 00:25:50 $
    }

#README
    tkiFileParseNode $fileKey {
File: builtin, Node: Info, Up: Top, Prev: About, Next: QuickHelp
Info Files
----------

tkInfo displays "info" files, a file format that supports a robust
hypercard system which is ideal for on-line help.  Each file contains
many nodes organized into a tree, and typically all the files on your
system are organized into a (flat) tree.  The top-level info file is
called "dir" or "dir.info", and lists all the other files on your
system (typically GNU software, since most GNU software is documented
using texinfo).

Info files can be created manually with any text editor (vi,emacs),
with the support of the emacs "info" package, with the GNU "makeinfo"
program, or with the emacs scripts for latexinfo.  The makeinfo program
produces a set of info files from TeX source that use the GNU texinfo
format (the one that uses "@" everywhere).  Similarly, the latexinfo
package (like texinfo, but with latex commands and syntax) provides
emacs scripts for producing info files.

There are several info file browsers or viewers:
  tkinfo  what you're using now, *note (builtin)About::.
  info    an ascii curses-based C program.
  emacs info mode: ??? (ask an emacs wizard, not me).
  xinfo   an X11 athena-based program, (old and unsupported).

See the texinfo package for the info viewer, makeinfo program,
emacs info mode and general information about info files.
All these info packages may be obtained by anonymous ftp from:
texinfo:	prep.ai.mit.edu		pub/gnu/texinfo-3.1.tar.z
latexinfo:	csc-sun.math.utah.edu	pub/tex/pub/latexinfo
xinfo:		gatekeeper.dec.com	pub/X11/contrib/xinfo.tar.Z
    }

    tkiFileParseNode $fileKey {
File: builtin, Node: QuickHelp, Up: Top, Prev: Info
		tkInfo Commands

Use scroll bar on right side to scroll through the node text.  Cross references
and menu entires are shown highlighted (blue foreground).  Press the left mouse
button over highlighted text to see that node.

Accelerator keys:
?	Show this quick help message.  Equivalent to `g(builtin)QuickHelp'.
h	Invoke the Info (emacs/tty) tutorial.  Equivalent to `g(info)Help'
n       Move to the "next" node of this node.
p       Move to the "previous" node of this node.
u       Move "up" from this node.
t	"Toggle" to last node you were at, toggle based.
l       Move to the "last" node you were at, stack based.
d       Move to the "directory" node.  Equivalent to `gDIR'.
D       Move to the builtin "directory" node.  Equivalent to `g(builtin)DIR'.
	If you have multiple info directories installed, use this key.
q	Quit info window.

Advanced commands:
1-9	Pick first, second, etc, item in node's menu.
g,(	Goto given node (Ctrl-g to abort,Ctrl-u to erase line).
	Syntax: NODENAME or (FILENAME)NODENAME.
s,/	Search for text literally or by regular expression.
Ctrl-s	Continue previous search.
m,f	menu,footnote (not implemented).
p	Print (not implemented).
!	Issue tcl command, results printed on stdout.

Scrolling commands:
b			Scroll to the beginning of the node.
e			Scroll to the end of the node.
SPACE, Ctrl-F,
Ctrl-V, PgDn		Scroll forwards one page.
BACKSPACE, DELETE,
Ctrl-B, Alt-V, PgUp	Scroll backwards one page.

Go "up" from this node to obtain more information on tkInfo.
    }

    tkiFileParseNode $fileKey {
File: builtin, Node: ToDo, Up: Top, Prev: QuickHelp
tkInfo is still incomplete.  The following is my todo list.

-  Better text widget bindings.  These are getting better with each
   release, but more are needed.  If you have suggestions, let me know.
-  Better menu bars, and more user-configurability (colors, fonts, etc).
-  Add option to allow all the "*note:" to not be drawn on the screen,
   or change them to "see also".
-  Implement all the currently unimplemented bindings.
-  Implement a tcl-only, tty-based interface?
-  Implement stat'ing of the source files with auto-reload.
-  Profile the whole mess: speed up file loading and node formating.
-  Figure out some huiristic for un-loading files to save memory.
-  Add in links to TkMan.
-  Extend searching to whole-file search, and make literal case-insen
   searching work right.  The coloring of search'd and hyper'd
   text when combined doesnt work right.
    }
}

proc _tkiFileFindSuf { fileName } {
    global tki

    foreach suf $tki(infoSuffix) {
	foreach extrasuf {"" .Z .z} {
	    set filePath "$fileName$suf$extrasuf"
	    if { [file isfile $filePath] } {
		return $filePath
	    }
	}
    }
    return ""
}

#
# Given {fileName} (see intro section above), find the cooresponding
# filepath.  The filepath of {pntFileKey}, if specified, is
# used as a starting point for locating {fileName}.
# Returns the file path if found, else empty string.
#
proc tkiFileFind { fileName {pntFileKey ""} } {
    global tki

    catch {unset tki(temp-search-path)}
    case [string index $fileName 0] {
      "/ . ~" {
	lappend tki(temp-search-path) [file dirname $fileName]
	# Should be valid UN*X path modulo suffix
	set filePath [_tkiFileFindSuf $fileName]
	if { "$filePath"!="" } { return $filePath }
	set filePath [_tkiFileFindSuf [string tolower $fileName]]
	if { "$filePath"!="" } { return $filePath }
      }
      default {
	# Try all the infopaths, and all suffixs
	set pp ""
	if { "$pntFileKey"!="" } {
	    set pp [file dirname [lindex $tki(fileinfo-$pntFileKey) 2]]
	}
	foreach prepath "$pp $tki(infoPath)" {
#puts stdout "Searching dir ``$prepath''"
	    lappend tki(temp-search-path) $prepath
	    if { ! [file isdir $prepath] } continue
	    set filePath [_tkiFileFindSuf $prepath/$fileName]
	    if { "$filePath"!="" } { return $filePath }
	    set filePath [_tkiFileFindSuf $prepath/[string tolower $fileName]]
	    if { "$filePath"!="" } { return $filePath }
	}
      }
    }
    return ""
}

#
# Given {fileName}, find the coorsponding filepath via tkiFindFile().
# Create a {fileKey} for the file, and make the appropriate table entries.
# Note that {fileName} must be just that, and not a filekey.
#
proc tkiFileAdd { fileName {pntFileKey ""} {wholeB 0} } {
    global tki

    if { [info exist tki(file-$fileName)] } {
	return $tki(file-$fileName)
    }
    set filePath [tkiFileFind $fileName $pntFileKey]
    if { "$filePath"=="" } { return "" }
    set fileKey fk[tkiGetSN]
    set tki(file-$fileName) $fileKey
    set tki(fileinfo-$fileKey)  [list $fileKey $fileName $filePath $pntFileKey $wholeB]
    set tki(incore-$fileKey) 0
    return $fileKey
}

proc tkiFileGet { fileSpec {pntFileKey ""} {wholeB 0} } {
    global tki

    if { [info exist tki(fileinfo-$fileSpec)] } {
	set fileKey $fileSpec
    } else {
	if { [info exist tki(file-$fileSpec)] } {
	    set fileKey $tki(file-$fileSpec)
	} else {
	    set fileKey [tkiFileAdd $fileSpec $pntFileKey $wholeB]
	    if { "$fileKey"=="" } {
		set msg "Can't locate info file ``$fileSpec''."
		if [info exist tki(temp-search-path)] {
		    set search $tki(temp-search-path)
		    regsub -all " " $search ", " search
		    append msg "\nSearch path is \{$search\}."
		}
		tkiError $msg
		return ""
	    }
	}
    }
    set fileinfo $tki(fileinfo-$fileKey)
    if { ! $tki(incore-$fileKey) } {
        tkiFileLoad $fileKey [lindex $fileinfo 1] [lindex $fileinfo 2] [lindex $fileinfo 4]
    }
    return $fileKey
}

proc _tkiFileLoadIndirectTbl { fileKey lines } {
    global tki

    set indirinfos ""
    foreach line $lines {
	if { "$line"!="" } {
	    set pair [split $line ":"]
	    if { [llength $pair] != 2 } {
		tkiFileWarning $fileKey "has bad file-indirect line ``$line''"
		continue
	    }
	    set indirKey [tkiFileAdd [lindex $pair 0] $fileKey]
	    if { "$indirKey"=="" } {
		tkiError "Can't locate indirect file ``[lindex $pair 0]''."
		continue
	    }
	    set byteOfs [string trim [lindex $pair 1]]
	    lappend indirinfos [list $indirKey $byteOfs]
	}
    }
    set tki(indirf-$fileKey) $indirinfos
#puts stdout "IndirectTbl $fileKey: $indirinfos"
}

proc _tkiFileLookupIndir { indirf byte } {
    set lastKey ""
    foreach fi $indirf {
	if { [lindex $fi 1] > $byte } break
	set lastKey [lindex $fi 0]
    }
    return $lastKey
}

proc _tkiFileLoadTagTbl { fileKey lines } {
    global tki

    set subkey [lindex $lines 0]
    if { "$subkey"!="(Indirect)" } return
    set indirf $tki(indirf-$fileKey)
    set indirinfos ""
    foreach line [lrange $lines 1 end] {
	if { "$line"=="" } continue
	set pair [split $line $tki(nodeByteSep)]
	if { [llength $pair] != 2 } {
	    tkiFileWarning $fileKey "has bad tag-indirect line ``$line''"
	    continue
	}
	set nodeName [string tolower [string trim [string range [lindex $pair 0] 5 end]]]
	set byteOfs [string trim [lindex $pair 1]]
	set indirFile [_tkiFileLookupIndir $indirf $byteOfs]
	lappend indirinfos [list $nodeName $byteOfs $indirFile]
# puts stdout "$fileKey: tag [list $nodeName $byteOfs $indirFile]"
    }
    set tki(indirn-$fileKey) $indirinfos
}

proc tkiFileParseNode { fileKey node } {
    global tki

    set lines [split $node "\n"]
    set keyline [string trim [lindex $lines 1]]
    case $keyline {
      { {[Ii]ndirect:} } {
	_tkiFileLoadIndirectTbl $fileKey [lrange $lines 2 end]
	return "IndirectTable"
      }
      { {[Tt]ag [Tt]able:} } {
	_tkiFileLoadTagTbl $fileKey [lrange $lines 2 end]
	return "TagTable"
      }
      { {[Ee]nd [Tt]ag [Tt]able} } {
	return "EndTagTable"
      }
    }
    # Some screwed up files omit the ``,'' for the file key.
    regsub "(File:\[^,\]*)Node:" $keyline "\\1,Node:" keyline
    set nodekey ""; set filekey ""
    set nextkey ""; set prevkey ""; set upkey ""
    foreach key [split $keyline ",\t"] {
	set key [string trim $key]
	case $key {
	  "File:*" { set filekey [string trim [string range $key 5 end]] }
	  "Node:*" { set nodekey [string trim [string range $key 5 end]] }
	  "Up:*"   { set upkey   [string trim [string range $key 3 end]] }
	  "Prev:*" { set prevkey [string trim [string range $key 5 end]] }
	  "Next:*" { set nextkey [string trim [string range $key 5 end]] }
	}
    }
    if { "$nodekey" == "" } { return "" }
    lappend tki(nodesinfo-$fileKey) [list [llength $tki(nodesinfo-$fileKey)] $nodekey $filekey $upkey $prevkey $nextkey]
    lappend tki(nodesbody-$fileKey) $node
    return $nodekey
}

proc tkiFileLoad { fileKey fileName filePath wholeB } {
    global tki

    case $filePath in {
      *.Z	{ set fp "|$tki(compresscat-Z) $filePath" }
      *.z	{ set fp "|$tki(compresscat-z) $filePath" }
      default	{ set fp $filePath }
    }
    if [catch {open $fp "r"} fid] {
	tkiError "Can't open ``$fp''."
	return ""
    }
    if { $wholeB } {
	set node_Top [list 0 Top $fileName "" "" ""]
    	set tki(nodesinfo-$fileKey) [list $node_Top]
    	set tki(nodesbody-$fileKey) [list [read $fid]]
        close $fid
        set tki(incore-$fileKey) 1
	return $fileKey
    }
    set nodelist [split [read $fid] $tki(nodeSep)]
    close $fid
    tkiStatus "Loading $fileName"
    set nodecnt 0
    set tki(nodesinfo-$fileKey) ""
    set tki(nodesbody-$fileKey) ""
    foreach node $nodelist {
	incr nodecnt
	if { $nodecnt==1 || [string length $node] < 10 } continue
	set nodeName [tkiFileParseNode $fileKey $node]
	if { "$nodeName" == "" } {
	    puts stdout "Warning: node #$nodecnt of file $filePath is bogus"
	    continue
	}
    }
    set tki(incore-$fileKey) 1
    return $fileKey
}

#
# Parse nodeSpec and fileSpec.  {nodeSpecVar} and {fileSpecVar} must
# refer to variables within the caller's context.  They will be substituted
# and replaced with canonical forms.
#
proc tkiParseNodeSpec { nodeSpecVar fileSpecVar } {
    global tki
    upvar $nodeSpecVar nodeSpec $fileSpecVar fileSpec

    if { "[string index $nodeSpec 0]"=="(" } {
	set ridx [string first ")" $nodeSpec]
	if { $ridx <= 0 } {
	    tkiError "Malformed nodespec ``$nodeSpec''."
	    return 0
	}
	set fileSpec [string range $nodeSpec 1 [expr $ridx-1]]
	set nodeSpec [string trim [string range $nodeSpec [expr $ridx+1] end]]
    }
    if { "$fileSpec"=="" } {
	set fileSpec $tki(topLevelFile)
    }
    if { "$nodeSpec"=="" } {
	set nodeSpec $tki(topLevelNode)
    }
    return 1
}

proc tkiFmtFileSpec { fileSpec } {
    global tki
    if [info exist tki(fileinfo-$fileSpec)] {
	return [lindex $tki(fileinfo-$fileSpec) 1]
    }
    return $fileSpec
}

proc tkiFmtNodeSpec { nodeSpec {fileSpec ""} } {
    global tki
    if ![tkiParseNodeSpec nodeSpec fileSpec] {
	return "Bad file/node spec ``$nodeSpec''"
    }
    set fileSpec [tkiFmtFileSpec $fileSpec]
    return "($fileSpec)$nodeSpec"
}

#
# This is the core search function.  It attempts to locate {nodeSpec}
# where ever it is.  {fileSpec} is a default file name that is used
# only if {nodeSpec} doesn't contain a reference.
# Returns a list {nodeIdx fileKey}, where {nodeIdx} is the index of the
# node within {fileKey}.
#
# As discussed in the intro above, at this level we cannot allow any concept of
# "current file" or "current node": it is up to the caller to maintain
# that information and pass up the appropriate arguments.
#
proc tkiGetNodeRef { nodeSpec {fileSpec ""} {pntFileKey ""} } {
    global tki

    if ![tkiParseNodeSpec nodeSpec fileSpec] {
	return ""
    }
    set wholeB 0
    if { "$nodeSpec"=="*" } {
	set wholeB 1
	set nodeSpec Top
    }
    set fileKey [tkiFileGet $fileSpec $pntFileKey $wholeB]
    if { "$fileKey"=="" } { return "" }
    set fileName [lindex $tki(fileinfo-$fileKey) 1]
    set realPntKey [lindex $tki(fileinfo-$fileKey) 3]
    tkiStatus "Searching for ``$nodeSpec'' in $fileName"
    set nodeSpec [string tolower $nodeSpec]

    # Popup to our indirect-parent, if it has a tag table
    if { "$pntFileKey"=="" && "$realPntKey"!="" 
      && [info exist tki(indirn-$realPntKey)] } {
	return [tkiGetNodeRef $nodeSpec $realPntKey]
    }

    #  Use index on this file, pushdown to our children
    if { [info exist tki(indirn-$fileKey)] } {
	# Use node index (indirect)
	foreach indir $tki(indirn-$fileKey) {
	    if { [string match $nodeSpec [lindex $indir 0]] } {
		set nodeRef [tkiGetNodeRef $nodeSpec [lindex $indir 2] $fileKey]
	        if { "$nodeRef"!="" } { return $nodeRef }
		tkiFileWarning $fileKey "Incorrect tag table"; break
	    }
	}
    }

    # Brute force on this file
    if { [info exist tki(nodesinfo-$fileKey)] } {
	foreach nodeinfo $tki(nodesinfo-$fileKey) {
	    if { [string match $nodeSpec [string tolower [lindex $nodeinfo 1]]] } {
		return [list [lindex $nodeinfo 0] $fileKey]
	    }
	}
    }

    # Look for node in all indirect files (brute force)
    if { [info exist tki(indirf-$fileKey)] } {
	foreach indir $tki(indirf-$fileKey) {
# puts stdout "Searching $indir"
	    set nodeRef [tkiGetNodeRef $nodeSpec [lindex $indir 0] $fileKey]
	    if { "$nodeRef"!="" } { return $nodeRef }
	}
    }

    # Look for node in my parent, but only if not called from my pnt
    if { "$pntFileKey"=="" } {
	if { "$realPntKey"!="" } {
	    return [tkiGetNodeRef $nodeSpec $realPntKey]
	}
    }
    return ""
}

#
# Initialize the regexp strings that are used later in 
# tkiNodeParseBody() (for xrefs) and tkiNodeParseMenu() (for menus).
# This func is called once from tkiInit() and then destroyed.
#
proc _tkiNodeParseInit { } {
    global tki

    # For xrefs, there are two forms:
    #	*note nodeSpec::terminator			(form 1)
    #   *note label: nodeSpec terminator		(form 2)
    # Terminator is ``.'' or ``,'', forms may wrap accross lines.
#old      "\*(note\[ \t\n\]*)(\[^:\]+)::"
    set tki(re_xref1_p) "\\*(note\[ \t\n\]*)(\[^:\]+)::"
    set tki(re_xref1_s) "x\\1\037c\\2\037dxx"
#old      "\*(note\[ \t\n\]*)(\[^:\]+)(:\[ \t\n\]*)(\(\[^ \t\n)\]+\))?(\[^.,\]*)\[.,\]"
    set tki(re_xref2_p) "\\*(note\[ \t\n\]*)(\[^:\]+)(:\[ \t\n\]*)(\\(\[^ \t\n)\]+\\))?(\[^.,\]*)\[.,\]"
    set tki(re_xref2_s) "x\\1\037a\\2\037b\\3\037c\\4\\5\037dx"


    # For menus, there are two forms:
    #	* nodeSpec::	comments...			(form 1)
    #   * label: nodeSpec[ \t.,] comments...		(form 2)
#old      "(\*\[ \t\]*)(\[^:\]+)::"
    set tki(re_menu1_p) "(\\*\[ \t\]*)(\[^:\]+)::"
    set tki(re_menu1_s) "\\1\037A\\2\037B"
    # rp2 = "* ws label: ws", rp2a="rp2 nodename ws", rp2b="rp2 (file)node ws"
#old      "(\*\[ \t\]*)(\[^:\]+)(:\[ \t\]*)(\(\[^ \t)\]+\))?(\[^\t.,\]*)"
    set tki(re_menu2_p) "(\\*\[ \t\]*)(\[^:\]+)(:\[ \t\]*)(\\(\[^ \t)\]+\\))?(\[^\t.,\]*)"
    set tki(re_menu2_s) "\\1\037A\\2\037B\\3\037C\\4\\5\037D"
}

#
# Parse a nody-body and identify the cross references.
#
proc tkiNodeParseBody { nodeName fileKey bodytext } {
    global tki

    set nl "node ``($fileKey)$nodeName''"
    regsub -all -nocase $tki(re_xref1_p) $bodytext $tki(re_xref1_s) bodytext
    regsub -all -nocase $tki(re_xref2_p) $bodytext $tki(re_xref2_s) bodytext
    set xrefinfo ""
    set curIdx 1
    foreach seg [split $bodytext "\037"] {
	set stIdx $curIdx
	set curIdx [expr { $curIdx + [string length $seg] - 1 }]
	if { "[string index $seg 0]"!="c" } continue
	set toNode [string range $seg 1 end]
#puts stdout "tkiNodeParseBody:1 ``$toNode''"
	regsub -all "\[ \t\n\]+" $toNode " " toNode
#puts stdout "tkiNodeParseBody:2 ``$toNode''"
	lappend xrefinfo [list [llength $xrefinfo] $toNode $stIdx $curIdx]
    }
    return $xrefinfo
}

#
# Parse the menu and extract the keywords
#
proc tkiNodeParseMenu { nodeName fileKey menutext } {
    global tki

    # There are two forms:
    #	* nodeSpec::	comments...			(form 1)
    #   * label: nodeSpec[ \t.,] comments...		(form 2)
    set rp1 $tki(re_menu1_p)
    set sp1 $tki(re_menu1_s)
    set rp2 $tki(re_menu2_p)
    set sp2 $tki(re_menu2_s)

    set menuinfo ""
    set linecnt 0; set menucnt 0
    set nl "node ``($fileKey)$nodeName''"
    foreach line [split $menutext "\n"] {
	incr linecnt
	if { "[string index $line 0]"!="*" 
	  || "[string range $line 0 6]"=="* Menu:" } continue
	incr menucnt
#puts stdout "Try rp $line"
	if { [regsub $rp1 $line $sp1 prsline] } {
	    set nBeg [expr { [string first "\037A" $prsline] + 0 } ]
	    set nEnd [expr { [string first "\037B" $prsline] - 3 } ]
	} else {
	    if { [regsub $rp2 $line $sp2 prsline] } {
	        set nBeg [expr { [string first "\037C" $prsline] - 4 } ]
	        set nEnd [expr { [string first "\037D" $prsline] - 7 } ]
	    } else {
		tkiFileWarning $fileKey "$nl has bad menu (line $linecnt)"
		continue
	    }
	}
	set toNode [string range $line $nBeg $nEnd]
	lappend menuinfo [list $linecnt $menucnt $toNode $nBeg $nEnd]
    }
    return $menuinfo
}

proc _tkiDpyWinAction { w action {extra ""} } {
    upvar #0 $w wvars
    global tki

    set toNode ""
    set toFile $wvars(fileKey) 
    case $action {
      quit {
	unset wvars
	destroy $w
	# XXX: !!This is a major hack!!
	global tkiEmbed
	if { ![info exist tkiEmbed] && "[winfo children .]"=="" } {
	    destroy .
	}
	return
      }
      goto {
	set wvars(searchB) 0
	set dd $w.s
	$dd.status conf -text [expr { "[string index $extra 0]"=="!" ? "Enter cmd: " : "Enter node: " } ]
	if { "$extra"=="" } { set extra $wvars(gotoStr) }
	_tkiWinGotoMap $w $extra
      }
      search {
	if { "$extra"=="incr" } {
	    set tki(curWindow) $w
	    searchboxNext searchkey $w.main.text $w.main.vsb
	    set tki(curWindow) ""
    	    $w.s.status conf -text $wvars(status)
	} else {
	    set wvars(searchB) 1
	    set wvars(searchRegexpB) [expr {"$extra"=="regexp"}]
	    #set wvars(searchCaseB) 0
	    $w.s.status conf -text "Search for: "
	    _tkiWinGotoMap $w $wvars(searchStr)
	}
      }
      last {
	set idx [expr { [llength $wvars(lastNodes)] - 2 } ]
	if { $idx >= 0 } {
	    set lastinfo [lindex $wvars(lastNodes) $idx]
	    set wvars(lastNodes) [lreplace $wvars(lastNodes) $idx end]
	    set toFile [lindex $lastinfo 0]
	    set toNode [lindex $lastinfo 1]
	}
      }
      toggle {
	if { [info exist wvars(toggle)] } {
	    set toFile [lindex $wvars(toggle) 0]
	    set toNode [lindex $wvars(toggle) 1]
	}
      }

      up   { set toNode [lindex $wvars(nodeinfo) 3] }
      prev { set toNode [lindex $wvars(nodeinfo) 4] }
      next { set toNode [lindex $wvars(nodeinfo) 5] }
      menu {
	if { [info exist wvars(menuinfo)] } {
	    if { "$extra"=="" } {
                $w.main.text yview menu.first
	    } else {
		set menuitem [lindex $wvars(menuinfo) $extra]
		set toNode [lindex $menuitem 2]
	    }
	}
      }
      scroll {
	set sbinfo [$w.main.vsb get]
	set windelta [expr {[lindex $sbinfo 1]-1}]
	set lastpage [expr {[lindex $sbinfo 0]-[lindex $sbinfo 1]}]
	case $extra {
	  forw { set gotoline [expr {[lindex $sbinfo 2]+$windelta}] }
	  back { set gotoline [expr {[lindex $sbinfo 2]-$windelta}] }
	  top { set gotoline 0 }
	  bottom { set gotoline $lastpage }
	}
	if { $gotoline > $lastpage } { set gotoline $lastpage }
	if { $gotoline < 0 } { set gotoline 0 }
	$w.main.text yview $gotoline
      }
    }
    if { "$toNode"!="" } {
        tkiDpyIdle $toNode $toFile $w
    }
}

proc _tkiWinGotoMap { w extra } {
    upvar #0 $w wvars
    set dd $w.s
    $dd.goto delete 0 end
    $dd.goto insert end $extra
    pack after $dd.status $dd.goto { left expand fill }
    if { $wvars(searchB) } {
	pack after $dd.goto $dd.regexp { left filly }
	pack after $dd.regexp $dd.case { left filly }
    }
    focus $dd.goto
}

proc _tkiWinGotoUnmap { w } {
    upvar #0 $w wvars
    set dd $w.s
    focus $w.main.text
    pack unpack $dd.goto
    pack unpack $dd.regexp
    pack unpack $dd.case
    $dd.status conf -text $wvars(status)
}

#
# This is called when <Return> is pressed in the "goto" text window.
# We could either be in a goto-node command, or a exec-tcl command.
# We take the appropriate action and cleanup.
#
proc _tkiWinGotoOk { w } {
    global tki
    upvar #0 $w wvars
    set dd $w.s
    set node [string trim [$dd.goto get]]
    if { $wvars(searchB) } {
	set wvars(searchB) 0
	set wvars(searchStr) $node
	set tki(curWindow) $w
	searchboxSearch $wvars(searchStr) $wvars(searchRegexpB) \
	  $wvars(searchCaseB) searchkey $w.main.text $w.main.vsb
	set tki(curWindow) ""
        _tkiWinGotoUnmap $w
    } else {
	set wvars(gotoStr) $node
	case $node {
	  "" { }
	  "!*" {
	    if [catch [string range $node 1 end] error] {
		puts stdout "Error: $error"
	    } else {
		puts stdout [expr { "$error"=="" ? "Ok" : "$error" }]
	    }
	    set node ""
	  }
	}
        _tkiWinGotoUnmap $w
	tkiDpyIdle $node $wvars(fileKey) $w
    }
}

proc _tkiWinGotoAbort { w } {
    upvar #0 $w wvars
    set wvars(searchB) 0
    _tkiWinGotoUnmap $w
}

#
# Note that its not safe to change the command of the buttons while
# the button press is active.  Thus we must use idle-time handler.
#
proc tkiMakeDpyWin { } {
    global tki

    while 1 {
        set w .tki[tkiGetSN]
	if { [catch {winfo parent $w}] } break
    }
    lappend tki(windows) $w
    upvar #0 $w wvars
    set wvars(nodeinfo) ""
    set wvars(fileKey) ""
    set wvars(infonodename) "(builtin)Top"
    set wvars(status) "Press ? for help."
    set wvars(gotoStr) ""
    set wvars(searchB) 0
    set wvars(searchStr) ""

    toplevel $w -class TkInfo
    wm title $w tkInfo
    wm iconname $w tkInfo
    wm minsize $w 20 20
    set dd $w.title; pack append $w [frame $dd] { top fillx }

    set dd $w.main; pack append $w [frame $dd] { top expand fill }
    pack append $dd [text $dd.text -state disabled -width 80 -wrap word] \
      { left expand fill }
    pack append $dd [scrollbar $dd.vsb -orient vert -com "$dd.text yview"] \
      { left fill }
    $dd.text config -yscroll "$dd.vsb set"

    set dd $w.div1; pack append $w [frame $dd -bd 1 -rel sunken \
      -height 3 -width 10] { top fillx }
    set dd $w.s; pack append $w [frame $dd] { top fillx }
    pack append $dd [label $dd.status -anc w] { left fillx }
#    place [label $dd.status -anc w] -in $dd -x 0 -y 0 -anc nw
#    pack append $dd [label $dd.forhelp -anc e -text "Press ? for help."] \
#      { right filly }
    place [label $dd.forhelp -anc e -text "Press ? for help."] \
      -in $dd -relx 1 -y 0 -anc ne
    entry $dd.goto -rel sunken
    checkbutton $dd.regexp -text "Regexp" -var ${w}(searchRegexpB)
    checkbutton $dd.case -text "Case Sen" -var ${w}(searchCaseB)
    bind $dd.goto <Return> "_tkiWinGotoOk $w"
    bind $dd.goto <Escape> "_tkiWinGotoAbort $w"
    bind $dd.goto <Any-Control-g> "_tkiWinGotoAbort $w"

    set dd $w.buts; pack append $w [frame $dd] { top fillx }
    pack append $dd [button $dd.quit -text "Quit Info" \
      -com "_tkiDpyWinAction $w quit"] { left exp fill }
    pack append $dd [button $dd.menu -text "Show Menu" \
      -com "_tkiDpyWinAction $w menu"] { left exp fill }
    pack append $dd [button $dd.prev -text "Prev Node" \
      -com "_tkiDpyWinAction $w prev"] { left exp fill }
    pack append $dd [button $dd.up   -text "Up Node" \
      -com "_tkiDpyWinAction $w up"] { left exp fill }
    pack append $dd [button $dd.next -text "Next Node" \
      -com "_tkiDpyWinAction $w next"] { left exp fill }
#    pack append $dd [button $dd.toggle -text "Toggle Node" \
#      -com "_tkiDpyWinAction $w back" -state disabled] { left exp fill }
    pack append $dd [button $dd.dup -text "New browser" \
      -com "tkiInfoWindow"] { left exp fill }
    set ddm $dd.options.m
    pack append $dd [menubutton $dd.options -text "Options..." \
      -menu $ddm -rel raised] { left exp fill }
    menu $ddm
    $ddm add check -lab "Show Headers" -var tki(rawHeadersB)
    $ddm add sep
    $ddm add com -lab "Node Look" -state disabled
    $ddm add radio -lab "Color"     -var tki(nodelook) -val color
    $ddm add radio -lab "Font"      -var tki(nodelook) -val font
    $ddm add radio -lab "Underline" -var tki(nodelook) -val underline

    set tw $w.main.text
    foreach win "$w.main.text $w.s.status" {
	bind $win <Key-q>	"$dd.quit invoke"
	bind $win <Key-m>	"$dd.menu invoke"
	bind $win <Key-p>	"$dd.prev invoke"
	bind $win <Key-u>	"$dd.up   invoke"
	bind $win <Key-n>	"$dd.next invoke"
	bind $win <Key-t>	"_tkiDpyWinAction $w toggle"
	bind $win <Key-l>	"_tkiDpyWinAction $w last"
	bind $win ?		[list tkiDpyIdle "(builtin)QuickHelp" "" $w]
	bind $win <Key-h>	[list tkiDpyIdle "(info)Help" "" $w]
	bind $win <Key-d>	[list tkiDpyIdle "(dir)Top" "" $w]
	bind $win <Key-D>	[list tkiDpyIdle "(builtin)Top" "" $w]
	bind $win <Key-g>	"_tkiDpyWinAction $w goto"
	bind $win (		"_tkiDpyWinAction $w goto ("
	bind $win !		"_tkiDpyWinAction $w goto !"
	bind $win <Key-s>	"_tkiDpyWinAction $w search text"
	bind $win /		"_tkiDpyWinAction $w search regexp"
	bind $win <Control-Key-s> "_tkiDpyWinAction $w search incr"
	bind $win <Key-Help>	"tkiContextHelp %W"
	bind $win <Key-1>	"_tkiDpyWinAction $w menu 0"
	bind $win <Key-2>	"_tkiDpyWinAction $w menu 1"
	bind $win <Key-3>	"_tkiDpyWinAction $w menu 2"
	bind $win <Key-4>	"_tkiDpyWinAction $w menu 3"
	bind $win <Key-5>	"_tkiDpyWinAction $w menu 4"
	bind $win <Key-6>	"_tkiDpyWinAction $w menu 5"
	bind $win <Key-7>	"_tkiDpyWinAction $w menu 6"
	bind $win <Key-8>	"_tkiDpyWinAction $w menu 7"
	bind $win <Key-9>	"_tkiDpyWinAction $w menu 8"
	bind $win b			"_tkiDpyWinAction $w scroll top"
	bind $win e			"_tkiDpyWinAction $w scroll bottom"
	bind $win <Key-space>		"_tkiDpyWinAction $w scroll forw"
	bind $win <Control-Key-f>	"_tkiDpyWinAction $w scroll forw"
	bind $win <Control-Key-v>	"_tkiDpyWinAction $w scroll forw"
	bind $win <Key-F35>		"_tkiDpyWinAction $w scroll forw"
	bind $win <Key-Delete>		"_tkiDpyWinAction $w scroll back"
	bind $win <Key-BackSpace>	"_tkiDpyWinAction $w scroll back"
	bind $win <Control-Key-b>	"_tkiDpyWinAction $w scroll back"
	bind $win <Alt-Key-v>		"_tkiDpyWinAction $w scroll back"
	bind $win <Key-F29>		"_tkiDpyWinAction $w scroll back"
    }

    bind $w <Any-Enter> "focus $w.main.text"
    bind $w <Any-Leave> "focus none"

    set tki(curWindow) $w
    return $w
}

#
# This is more subtle than one might think.  Note that the text index
# "+1line" wont work on the last line of text, because the newline is
# considered part of the previous line.  Thus we use "lineend" instead.
#
proc _tkiTextTrim { w idx } {
    while 1 {
	set nidx [$w index "$idx lineend"]
	if { "[string trim [$w get $idx $nidx]]"!="" || "[$w index end]"=="1.0" } break
	$w delete $idx "$nidx +1char"
    }
}


# Modified version of ouster's version
proc _tkiTextInsertWithTags { w index text args } {
    set start [$w index $index]
    $w insert $start $text
    foreach tag $args {
    	$w tag add $tag $start insert
    }
}

proc _tkiConfChainButton { w which toNode } {
    $w.buts.$which conf -state [expr { "$toNode"=="" ? "disabled" : "normal" } ]
}

proc _tkiNodeLookTag { tw tag } {
    global tki
    case $tki(nodelook) {
      color { $tw tag conf $tag -fore $tki(nodelookColor) }
      underline { $tw tag conf $tag -underline 1 }
      font { $tw tag conf $tag -font $tki(nodelookFont) }
    }
}

proc _tkiDpyNodeBody { w nodeName nodeIdx fileKey bodytext } {
    global tki

    set tw $w.main.text
    $tw insert end $bodytext

    if { [info exist tki(xrefinfo-$fileKey-$nodeIdx)] } {
	set xrefinfo $tki(xrefinfo-$fileKey-$nodeIdx)
    } else {
        set xrefinfo [tkiNodeParseBody $nodeName $fileKey $bodytext]
	set tki(xrefinfo-$fileKey-$nodeIdx) $xrefinfo
    }
    set ms "1.0"
    $tw tag delete xrefkey
    foreach xi $xrefinfo {
	# xi = { xrefidx toNode startIdx endIdx }
	set xrefidx [lindex $xi 0]
	set toNode [lindex $xi 1]
        $tw tag add xrefkey "$ms +[lindex $xi 2] c" "$ms +[lindex $xi 3] c"
        $tw tag add xref$xrefidx "$ms +[lindex $xi 2] c" "$ms +[lindex $xi 3] c"
	$tw tag bind xref$xrefidx <ButtonRelease-1> [list tkiDpyIdle $toNode $fileKey $w]
    }
    _tkiNodeLookTag $tw xrefkey

    _tkiTextTrim $tw 1.0
    if { ! $tki(rawHeadersB) } {
	$tw delete 1.0 "1.0 +1line"
        _tkiTextTrim $tw 1.0
    }
}

proc _tkiDpyNodeMenu { w nodeName nodeIdx fileKey menutext } {
    global tki
#puts stdout "_tkiDpyNodeMenu: $nodeName"
    if { [info exist tki(menuinfo-$fileKey-$nodeIdx)] } {
	set menuinfo $tki(menuinfo-$fileKey-$nodeIdx)
    } else {
        set menuinfo [tkiNodeParseMenu $nodeName $fileKey $menutext]
	set tki(menuinfo-$fileKey-$nodeIdx) $menuinfo
    }
    set tw $w.main.text
    _tkiTextInsertWithTags $tw end $menutext menu
    $tw tag delete menukey
    foreach mi $menuinfo {
	# mi = { lineidx menuidx toNode nBeg nEnd }
	set lineidx [lindex $mi 0]
	set menuidx [lindex $mi 1]
	set toNode [lindex $mi 2]
	set ms "menu.first +$lineidx lines -1 lines"
	$tw tag add menukey "$ms +[lindex $mi 3] c" "$ms +[lindex $mi 4] c +1 c"
	$tw tag add menu$menuidx "$ms linestart" "$ms lineend"
	$tw tag bind menu$menuidx <ButtonRelease-1> [list tkiDpyIdle $toNode $fileKey $w]
#	$tw tag bind menu$menuidx <Key> {puts stdout "key %A press"}
    }
    _tkiNodeLookTag $tw menukey
}

proc _tkiDisplayNodeCore { w fileKey info body } {
    global tki; upvar #0 $w wvars

    set wvars(fileKey) $fileKey
    set wvars(nodeinfo) $info
    set nodeIdx [lindex $info 0]
    set nodeSpec "([lindex $info 2])[lindex $info 1]"
    tkiStatus "Formating $nodeSpec"
    set tw $w.main.text
    $tw conf -cursor $tki(waitCursor)
    set menuidx -1
    if { ! [lindex $tki(fileinfo-$fileKey) 4] } {
	set menuidx [string first "\n* Menu:" $body]
	if { $menuidx > 0 } {
	    set menutext [string range $body [expr {$menuidx+1}] end]
	    set body [string range $body 0 $menuidx]
	}
    }

    #
    # Config the up,prev,next headers
    #
    set nodeName [lindex $info 1]
    _tkiConfChainButton $w up   [lindex $info 3]
    _tkiConfChainButton $w prev [lindex $info 4]
    _tkiConfChainButton $w next [lindex $info 5]

    #
    # Config the main section
    #
    $tw conf -state normal
    $tw delete 1.0 end
    _tkiDpyNodeBody $w $nodeName $nodeIdx $fileKey $body
    if { [info exist menutext] } {
	_tkiDpyNodeMenu $w $nodeName $nodeIdx $fileKey $menutext
	set wvars(menuinfo) $tki(menuinfo-$fileKey-$nodeIdx)
	$w.buts.menu conf -state normal
    } else {
	catch {unset wvars(menuinfo)}
	$w.buts.menu conf -state disabled
    }
    $tw mark set insert 1.0
    $tw mark set anchor insert
    $tw tag remove sel 1.0 end
    $tw conf -state disabled -cursor $tki(normCursor)
#   set wvars(status) "$nodeSpec                           Press ? for help."
    set wvars(status) "$nodeSpec"
    wm title $w "tkInfo      $nodeSpec"
    tkiStatus $wvars(status)

    # Do search colors
    $tw tag conf searchkey -foreground [lindex [$tw conf -background] 4] \
      -background [lindex [$tw conf -foreground] 4]

    # This is really gross
    focus $tw
    after 1 [list $tw tag remove sel 1.0 end]
}


proc tkiDisplayNode { w fileKey info body } {
    global tki; upvar #0 $w wvars

    if { [info exist wvars(nodeinfo)] } {
	set wvars(toggle) [list $wvars(fileKey) [lindex $wvars(nodeinfo) 1]]
#        $w.buts.toggle conf -state normal
    }
    lappend wvars(lastNodes) [list $fileKey [lindex $info 1]]

    _tkiDisplayNodeCore $w $fileKey $info $body
}


proc tkiGetAndDisplay { nodeSpec {fileSpec ""} {w ""} } {
    global tki
    if { ! [info exist tki] } { tkiInit }
    if { "$w"=="" } {
	set w [tkiMakeDpyWin]
    }
    set tki(curWindow) $w
#puts stdout "tkiGetAndDisplay: spec $nodeSpec"
    set nodeRef [tkiGetNodeRef $nodeSpec $fileSpec]
    if { "$nodeRef"=="" } {
	set fmtSpec [tkiFmtNodeSpec $nodeSpec $fileSpec]
	tkiError "Can't locate info node ``$fmtSpec''"
        set tki(curWindow) ""
	return ""
    }
#puts stdout "tkiGetAndDisplay: ref $nodeRef"
    set nodeIdx [lindex $nodeRef 0]
    set fileKey [lindex $nodeRef 1]
    tkiDisplayNode $w $fileKey [lindex $tki(nodesinfo-$fileKey) $nodeIdx] \
      [lindex $tki(nodesbody-$fileKey) $nodeIdx]
    set tki(curWindow) ""
    return $nodeRef
}

proc tkiReDpyWin { w } {
    global tki; upvar #0 $w wvars
    if ![info exist wvars(nodeinfo)] return
    set nodeinfo $wvars(nodeinfo)
    tkiGetAndDisplay [lindex $nodeinfo 1] $wvars(fileKey) $w
}

proc _tkiDpyIdle_Do { nodeSpec fileSpec w } {
    global tki

#puts stdout "tkiDpyIdle_Do: $nodeSpec"
    update
    set tw $w.main.text
    tkiGetAndDisplay $nodeSpec $fileSpec $w
    $tw conf -cursor $tki(normCursor)
}

proc tkiDpyIdle { nodeSpec {fileSpec ""} {w ""} } {
    global tki
    if { "$w"!="" } {
	set tw $w.main.text
	if { "[lindex [$tw conf -cursor] 4]"=="$tki(waitCursor)" } return
	$tw conf -cursor $tki(waitCursor)
    }
#puts stdout "tkiDpyIdle: $nodeSpec"
    after 1 [list _tkiDpyIdle_Do $nodeSpec $fileSpec $w]
}

proc tkiInfo { nodeSpec } {
    tkiGetAndDisplay $nodeSpec
    return ""
}

#
# Do stand-alone help window
# The -node option is for compatibility to the info program only.
#
proc tkiInfoWindow { args } {
    global tki

    if { ! [info exist tki] } { tkiInit }

    set w ""
    set nodeSpec ""
    set fileSpec ""
    set fileSpec2 ""
    set dirList ""
    set nodelook ""
    set headersB -1
    set opt_list { 
      { "window" w } 
      { "dir" dirList append } 
      { "file" fileSpec }
      { "headers" headersB bool }
      { "nodelook" nodelook }
      { "infofile" fileSpec2 }
      { "node" nodeSpec append }
    }

    set args [topgetopt $opt_list $args]

    if { "$dirList"!="" } {
	set infoPaths ""
	foreach dir $dirList {
	    eval lappend infoPaths [split $dir ":"]
	}
	tkiAddInfoPaths $infoPaths
    }
    if { "$nodelook" != "" }	{ set tki(nodelook) $nodelook }
    if { $headersB != -1 }	{ set tki(rawHeadersB) $headersB }
    if { "$fileSpec"=="" }	{ set fileSpec $fileSpec2 }
    if { "$fileSpec"!="" }	{ tkiAddInfoPaths [file dirname $fileSpec] }

    if { "$args"!="" } {
	eval lappend nodeSpec $args
    }
    if { [llength $nodeSpec] > 1 } {
	error "tkiInfoWindow: Only one node may be specified"
    }
    tkiGetAndDisplay [lindex $nodeSpec 0] $fileSpec $w
    return ""
}

#
# Start at window {w}, and traverse up the window tree looking for a variable
# of the form "$w(infonodename)".  If found, a window displaying that node
# will be generated.  {fileSpec} may be used to augment the infonode,
# and {infowin} may specific a pre-existing info window returned by
# tkiGetAndDisplay().
# 
proc tkiContextHelp { w {fileSpec ""} {infowin ""} } {
    for {} {"$w"!=""} {set w [winfo parent $w]} {
	# Line below is kludgy, b/c I cant see any other way to do it.
	if [uplevel #0 [list info exist ${w}(infonodename)]] {
	    upvar #0 $w wvars 
    	    return [tkiGetAndDisplay $wvars(infonodename) $fileSpec $infowin]
	}
    }
    if { "$fileSpec"!="" } {
    	return [tkiGetAndDisplay Top $fileSpec $infowin]
    }
    return [tkiGetAndDisplay (builtin)QuickHelp "" $infowin]
}

#
# We are operating in one of two modes:
#   1)  Stand-alone.  Popup an initial window, filling it according to argv.
#	Kill the stupid "." window.
#   2)	Embedded within a larger application.  Don't do anything automatically;
#	instead, let that application's startup script handle things.
#
# We are operating in embedded mode iff the global tkiEmbed exists.
#
proc tkiBoot { } {
    global argv tkiEmbed

    if { [info exist tkiEmbed] } return
    wm withdraw .
    if { "[lindex $argv 0]"!="" && [file isfile [lindex $argv 0]] } {
	# Some wishs pass the filename as argv[0].  Kill it off.
	set argv [lreplace $argv 0 0]
    }
    eval tkiInfoWindow $argv
}

tkiReset
tkiBoot
