/* 
  libSzarp - SZARP library

*/
/*
 * $Id$
 */
#ifndef _MBRTU_H_
#define _MBRTU_H_

#include "modbus.h"

#define MAX_BODY_SIZE 255

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#define IN_BUF 1
#define OUT_BUF 2
#define BOTH_BUF 3

/*
 * Parametry komunikacyjne 
 */
#define NO_PARITY 0
#define EVEN 1
#define ODD 2

/*
 * Kody bledow 
 */
#define CRC_ERROR -1
#define CRC_OK 1
/*
 * Kod bdw select-a
 */
#define GENERAL_ERROR -1
#define TIMEOUT_ERROR -2 /* Bd kiedy nie przyjd dane */

/*
 * Maksymalna wieko bufora pakietw
 */
#define MAX_BUFFER_SIZE 1000

/*
 * Tablica mwica ktre parametry w ramce MASTER s parzyste
 */
#define NSF -1 //Znacznik Not Supported Function
#define PARITY 0
#define INPARITY 1

 /* Tablica jest indeksowana nuerami funkcji */
 
const  int  MasterParity [] = {NSF, //00
	  		    NSF, //01
			    NSF, //02
			    PARITY, //03
			    PARITY, //04
			    PARITY, //05	
  			    PARITY, //06
  			    NSF,  //07
  			    NSF,  //08
  			    NSF,  //09
  			    NSF,  //10
  			    NSF,  //11
                            NSF,  //12
  			    NSF,  //13
  			    NSF,  //14
  			    NSF,  //15
                            INPARITY}; //16

/*
 * Tablica mwica ktre parametry w ramce SLAVE s parzyste
 */
const  int  SlaveParity [] = {NSF, //00
	  		    NSF, //01
			    NSF, //02
			    INPARITY, //03
			    INPARITY, //04
			    INPARITY, //05	
			    INPARITY, //06
  			    NSF,  //07
  			    NSF,  //08
  			    NSF,  //09
  			    NSF,  //10
  			    NSF,  //11
                            NSF,  //12
  			    NSF,  //13
  			    NSF,  //14
  			    NSF,  //15
                            PARITY}; //16

extern	char            mbrtu_single;
/** Struktura zawierajca budow ramki master */
	typedef struct {
		unsigned char   DeviceId;
			 /**<Numer identyfikacyjny jednostki podrzdnej*/
		unsigned char   FunctionId;
			   /**<Numer funkcji */
		unsigned short  Address;
			 /**<Adres rejestru do odczytu */
		unsigned short  Body[MAX_BODY_SIZE];
				      /**<Cialo pakietu w postaci gotowych rejestrow*/
		unsigned short  DataSize;
			  /**<Ilo przesyanych sw (Sowo jest dwubajtowe) */
	} tWMasterFrame;

/** Struktura zawierajca budowe ramki slave */
	typedef struct {
		unsigned char   DeviceId;
			 /**<Numer identyfikacyjny jednostki podrzednej */
		unsigned char   FunctionId;
			   /**<Numer funkcji */
		unsigned short  DataSize;
			  /**<Ilosc przesylanych bajtow */
		unsigned short  Address;
			 /**<Adres rejestru do odczytu */
		unsigned short  Body[MAX_BODY_SIZE];
				      /**<Cialo pakietu w postaci gotowych rejestrow*/
	} tWSlaveFrame;

/** Struktura zawierajca budow odczytanej ramki MASTER */
	typedef struct {
		unsigned char   DeviceId;
			/**<Numer identyfikacyjny jednostki podrzednej */
		unsigned char   FunctionId;
			   /**<Numer funkcji */
		unsigned short  Address;
			 /**<Adres rejestru do odczytu*/
		unsigned short  DataSize;
			 /**<Wielko danych */
		unsigned short  Body[MAX_BODY_SIZE];
				     /**<Cialo pakietu w postaci gotowych rejestrow*/
		unsigned short  CRC;
		     /**<Suma kontrolna CRC*/
		unsigned short  PacketSize;
			    /**<Rzeczywista wielko pakietu*/
	} tRMasterFrame;

/** Struktura zawierajca budow odczytanej ramki SLAVE */
	typedef struct {
		unsigned char   DeviceId;
			/**<Numer identyfikacyjny jednostki podrzednej */
		unsigned char   FunctionId;
			   /**<Numer funkcji */
		unsigned short  Address;
			 /**<Adres rejestru do odczytu*/
		unsigned short  DataSize;
			  /**<Wielko danych */
		unsigned short  Body[MAX_BODY_SIZE];
				    /**<Cialo pakietu w postaci gotowych rejestrow*/
		unsigned short  CRC;
		    /**<Suma kontrolna CRC*/
		unsigned short  PacketSize;
			   /**<Rzeczywista wielko pakietu*/
	} tRSlaveFrame;

/** Struktura zawierajca budow ramki error */
	typedef struct {
		unsigned char   DeviceId;
			 /**<Numer identyfikacyjny jednostki podrzdnej*/
		unsigned char   FunctionId;
			   /**<Numer funkcji */
		unsigned short  Exception;
			   /**<Kod bdu */
	} tWErrorFrame;

/**
 * Funkcja otwiera port szeregowy
 * @param Device Scieka do urzdzenia np "/dev/ttyS0"
 * @param BaudRate Prdko
 * @param StopBits bity stopu 1 albo 2
 * @param Parity Parzysto : NO_PARITY, ODD, EVEN 
 * @param return numer identyfikacyjny urzadzenia
 */
	int             InitComm(const char *Device, long BaudRate,
				 unsigned char DataBits,
				 unsigned char StopBits,
				 unsigned char Parity);

/**
 * Funkcja czyci bufory wejcia/wyjcia
 * @param CommId identyfikator portu
 * @param Mode tryb czyszczenia IN_BUF - bufor wejcia, OUT_BUF - bufor wyjcia, BOTH_BUF - obaq bufory
 */

	void            ClearPort(int CommId, int Mode);

/**
 * Funkcja Liczy sume CRC16 wg standart MODBUS-a 
 * @param Packet cay pakiet (dane)
 * @param PacketSize Ilo danych z ktrych ma by policzone CRC
 * @param return Warto CRC16
 */
	unsigned short  CRC16(unsigned char *Packet, int PacketSize);

/**
 * Funkcja konwertuje int na postac binarna
 * @param iint Wartosc int do konwersji
 * @param bMSB Bardziej znaczace slowo po konwersji na postac binarna
 * @param bLSB Mniej znaczace slowo po konwersji na postac binarna
 */

	void            int2bin(signed short iint, unsigned short *oBIN);

/**
 * Funkcja konwertuje postac binarna na int
 * @param iBIN warto binarna do konwersji na posta int
 * @param oint Warto int po konwersji
 */
	void            bin2int(unsigned short iBIN, signed short *oint);

/**
 * Funkcja konwertuje float na postac binarna
 * @param ifloat Wartosc zmiennoprzecinkowa do konwersji
 * @param bMSB Bardziej znaczace slowo po konwersji na postac binarna
 * @param bLSB Mniej znaczace slowo po konwersji na postac binarna
 */


	void            float2bin(float ifloat, unsigned short *bMSB,
				  unsigned short *bLSB);

/**
 * Funkcja konwertuje postac binarna na float
 * @param bMSB Bardziej znczace sowo do konwersji na posta flo
 * @param bLSB Mniej znaczce sowo do konwersji na posta float
 * @param ofloat Warto zmiennoprzecinkowa po konwersji
 */
	void            bin2float(unsigned short bMSB, unsigned short bLSB,
				  float *ofloat);

/**
 * Funkcja konwertuje floata na inta z przecinkiem
 * @param ifloat float do konwersji
 * @param prec Precyzja int-a
 * @param return Przekonwertowana warto int
 */
	signed short    float2int(float ifloat, unsigned short prec);

/**
 * Funkcja konwertuje floata na inta4 z przecinkiem
 * * @param ifloat float do konwersji
 * @param prec Precyzja int-a
 * @param return Przekonwertowana warto int4
 */
	signed int  float2int4(float ifloat, unsigned short prec);

	
/**
 * Funkcja konwertuje inta z przecinkiem na floata
 * @param iint int do konwersji
 * @param prec Precyzja int-a
 * @param return Przekonwertowana warto float
 */
	float           int2float(signed short iint, unsigned short prec);

/**
 * Converts short int to BCD format.
 * @param val value to convert
 * @param ret_code buffer for return code, 0 for success, 1 for error (may be not NULL)
 * @return BCD representation of int, undefined in case of error
 */
unsigned short int2bcd(unsigned short val, int* ret_code);

/**
 * Converts from BCD format to short int.
 * @param val value to convert
 * @param ret_code buffer for return code, 0 for success, 1 for error (may be not NULL)
 * @return integer value of converted val, undefined in case of error
 */
unsigned short bcd2int(unsigned short val, int* ret_code);

/**
 * Funkcja tworzy ramke typu MASTER
 * @param MasterFrame Struktura definiujaca parametry ramki
 * @param oPacket Pakiet przygotowany do wysania
 * @param MaxPacketSize Maksymalna wielko pakietu
 * @param PacketSize Wielko pakietu
 */
	void            CreateMasterPacket(tWMasterFrame MasterFrame,
					   unsigned char *oPacket,
					   int MaxPacketSize,
					   int *PacketSize);

/**
 * Funkcja dekoduje ramk MASTER do bardziej czytelnej postaci
 * @param *iPacket wskanik do bufora w ktrym znajduje si pakiet przychodzcy
 * @param MaxPacketSize - Maksymalna wielko pakietu
 * @param *MasterFrame - Wskanik do struktury zawierajcej zdekodowane dane
 */
	unsigned short  DecodeMasterPacket(unsigned char *iPacket,
					   int MaxPacketSize,
					   tRMasterFrame * MasterFrame);
/**
 * Funkcja tworzy ramk typu SLAVE
 * @param SlaveFrame - struktura definiujca parametry ramki
 * @param oPacket - Pakiet przygotowany do wysania
 * @param MaxPacketSize - Maksymalna wielko pakietu
 * @param PacketSize - Wielko pakietu
 */
	void            CreateSlavePacket(tWSlaveFrame SlaveFrame,
					  unsigned char *oPacket,
					  int MaxPacketSize, int *PacketSize);

/**
 * Funkcja tworzy ramk typu SLAVE
 * @param ErrorFrame - struktura definiujca parametry ramki
 * @param oPacket - Pakiet przygotowany do wysania
 * @param MaxPacketSize - Maksymalna wielko pakietu
 * @param PacketSize - Wielko pakietu
 */
	void            CreateErrorPacket(tWErrorFrame ErrorFrame,
					  unsigned char *oPacket,
					  int MaxPacketSize, int *PacketSize);

/**
 * Funkcja dekoduje ramk SLAVE do bardziej czytelnej postaci
 * @param *iPacket wskanik do bufora w ktrym znajduje si pakiet przychodzcy
 * @param MaxPacketSize - Maksymalna wielko pakietu
 * @param *SlaveFrame - Wskanik do struktury zawierajcej zdekodowane dane
 */

	unsigned short  DecodeSlavePacket(unsigned char *iPacket,
					  int MaxPacketSize,
					  tRSlaveFrame * SlaveFrame);


/**
 *  Funkcja wysyajca i odbierajca dane z urzdzenia master
 * @param CommId - numer identyfikacyjny urzadzenia 
 * @param iMasterFrame - Ramka do wyslania Master
 * @param return - kod bdu
 */
	int     SendMasterPacket(int CommId,
					 tWMasterFrame iMasterFrame);

/**
 *  Funkcja odbierajca pakiet slave
 * @param CommId - numer identyfikacyjny urzadzenia 
 * @param iSlaveFrame - Ramka do wysania Slave
 * @param *CRCStatus - stan CRC CRC_OK | CRC_ERROR (moze byc NULL)
 * @param ReceiveTimeout - timeout dla select-a
 * @param DelayBetweenChars - opoznienie pomiedzy przychodzacymi znakami
 * @param return - Kod bdu
 */
	int     ReceiveSlavePacket(int CommId,
					   tRSlaveFrame * oSlaveFrame,
					   char *CRCStatus,
					 int ReceiveTimeout, 
					 int DelayBetweenChars);
/**
 *  Funkcja wysyajca pakiet slave do urzdzenia master
 * @param CommId - numer identyfikacyjny urzadzenia 
 * @param iSlaveFrame - Ramka do wysania Slave
 * @param return - Kod bdu
 */
	int           SendSlavePacket(int CommId, tWSlaveFrame iSlaveFrame);

/**
 *  Funkcja wysyajca pakiet slave do urzdzenia master
 * @param CommId - numer identyfikacyjny urzadzenia 
 * @param iSlaveFrame - Ramka do wysania Slave
 * @param ReceiveTimeout - timeout dla select-a
 * @param DelayBetweenChars - opoznienie pomiedzy przychodzacymi znakami
 * @param *CRCStatus - stan CRC CRC_OK | CRC_ERROR (moze byc NULL)
 * @param ReceiveTimeout - timeout dla select-a
 * @param DelayBetweenChars - opoznienie pomiedzy przychodzacymi znakami
 * @param return - Kod bdu
 */
	int     ReceiveMasterPacket(int CommId,
					    tRMasterFrame * oMasterFrame,
					    char *CRCStatus,
					    int ReceiveTimeout, 
					    int DelayBetweenChars);


/**
 *  Funkcja wysyajca pakiet error do urzdzenia master
 * @param CommId - numer identyfikacyjny urzadzenia 
 * @param iErrorFrame - Ramka do wysania error
 */
	void            SendErrorPacket(int CommId, tWErrorFrame iErrorFrame);

/**
 * Funkcja zamyka poczenie i przywraca dawne ustawienia
 * @param CommId - numer identyfikacyjny urzadzenia
*/
	void            CloseComm(int CommId);
#endif
