/*
 * xconvers - GTK+ convers client for amateur radio
 * Copyright (C) 2000-2001 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * callback.c - functions that connect to a widget.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "history.h"
#include "preferences.h"
#include "net.h"
#include "color.h"
#include "types.h"

/************************ DEFINES AND GLOBAL VARIABLES ***********************/

#define HOSTNAMEHISTORY 10
#define PORTHISTORY     10

extern GtkWidget *mainwindow;
gchar *preferencesdir, *logfilename;
gpointer cvhost, cvport;
GdkColor white, grey, red, pink, color[10];
GdkFont *textfont;
GtkWidget *preferencesdialog;
gint rxmonitor;
gboolean connected = FALSE, locked = FALSE;
GList *txhistory, *hostnamehistory, *porthistory;
FILE *logfp;
preferencestype preferences;

/**********************************MAIN WINDOW********************************/

/*
 * This function is called when the main window is created. It is used to set
 * the background, font and the different colors for the user messages. It also
 * calls loadpreferences for retrieving default settings, sets a greeting and
 * the window title.
 */

void on_mainwindow_show(GtkWidget *widget, gpointer user_data)
{
  GtkWidget *menuclose, *maintext, *mainstatusbar, *mainscrolledwindow;
  GString *greeting;
  GtkStyle *maintextstyle;

  dircheck();
  loadpreferences();
  loadhistory();
  menuclose = lookup_widget(mainwindow, "close");
  gtk_widget_set_sensitive(menuclose, 0);
  gdk_colormap_alloc_color(gdk_colormap_get_system(), 
    &preferences.background, 1, 1);
  gdk_color_parse("white", &white);
  gdk_color_parse("red", &red);
  gdk_color_parse("pink", &pink);
  gdk_color_parse("light grey", &grey);
  gdk_color_parse("yellow", &color[0]);
  gdk_color_parse("green", &color[1]);
  gdk_color_parse("blue", &color[2]);
  gdk_color_parse("magenta", &color[3]);
  gdk_color_parse("orange", &color[4]);
  gdk_color_parse("cyan", &color[5]);
  gdk_color_parse("purple", &color[6]);
  gdk_color_parse("DarkKhaki", &color[7]);
  gdk_color_parse("DarkCyan", &color[8]);
  gdk_color_parse("DeepPink", &color[9]);
  maintext = lookup_widget(mainwindow, "maintext");
  maintextstyle = gtk_style_new();
  maintextstyle->base[GTK_STATE_NORMAL] = preferences.background;
  maintextstyle->base[GTK_STATE_INSENSITIVE] = preferences.background;
  gtk_widget_set_style(maintext, maintextstyle);
  textfont = gdk_font_load(preferences.font->str);
  if (!preferences.menu) widget_visible("mainmenubar", 0);
  if (!preferences.statusbar) widget_visible("mainstatusbar", 0);
  if (!preferences.scrollbar)
  {
    mainscrolledwindow = lookup_widget(mainwindow, "mainscrolledwindow");
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainscrolledwindow), 
      GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  };
  greeting = g_string_new("Welcome to ");
  g_string_append(greeting, PACKAGE);
  mainstatusbar = lookup_widget (mainwindow, "mainstatusbar");
  gtk_statusbar_push (GTK_STATUSBAR(mainstatusbar), 1, greeting->str);
  if (!preferences.statusbar)
  {
    g_string_prepend(greeting, "--> ");
    g_string_append(greeting, "\n");
    maintext_add(greeting->str, MESSAGE_TX);
  }
  g_string_free(greeting, TRUE);
}

/*
 * Try to catch PgUp/PgDn key so the receive window can scroll. Up and Down
 * arrow will show command line history. Alt key will activate menu. Typing
 * any other key will give focus to the transmit widget.
 */

gboolean on_mainwindow_key_press_event(GtkWidget *widget, GdkEventKey *event,
  gpointer user_data)
{
  GtkWidget *maintext, *mainmenubar;
  GtkAdjustment *adj;

  maintext = lookup_widget(mainwindow, "maintext");
  adj = GTK_ADJUSTMENT(GTK_TEXT(maintext)->vadj);
  switch (event->keyval)
  {
    case GDK_Up:
      gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
      tx_previous();
    break;
    case GDK_Down:
      gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
      tx_next();
    break;
    case GDK_Page_Up:
      gtk_adjustment_set_value(adj,adj->value-adj->page_size);
    break;
    case GDK_Page_Down:
      if(adj->value < adj->upper - adj->page_size)
        gtk_adjustment_set_value(adj, adj->value+adj->page_size);
    break;
    case GDK_Alt_L:
      mainmenubar = lookup_widget(mainwindow, "mainmenubar");
      gtk_widget_grab_focus(mainmenubar);
    break;
    default:
    break;
  }
  return FALSE;
}

/*
 * When a message is entered into the one-line entry at the bottom of the main
 * window, this function is called. The whole entry is grabbed, sent to the 
 * socket and cleared again.
 */

void on_mainentry_activate(GtkEditable *editable, gpointer user_data)
{
  GString *entry;
  GtkWidget *mainentry;

  mainentry = lookup_widget(mainwindow, "mainentry");
  entry = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(mainentry), 0, -1));
  gtk_entry_select_region(GTK_ENTRY(mainentry), 0, 
    GTK_ENTRY(mainentry)->text_length);
  tx(entry);
  g_string_free(entry, TRUE);
  gtk_entry_set_text(GTK_ENTRY(mainentry),"");
  gtk_widget_grab_focus(GTK_WIDGET(mainentry));
}

/*
 * Cleanup, save history and exit the gtk_main loop nicely if the main window
 * is deleted.
 */

gboolean on_mainwindow_delete_event(GtkWidget *widget, GdkEvent *event,
  gpointer user_data)
{
  savehistory();
  if (txhistory) g_list_free(txhistory);
  if (hostnamehistory) g_list_free(hostnamehistory);
  if (porthistory) g_list_free(porthistory);
  gtk_exit(0);
  return FALSE;
}

/********************************* MENUS *************************************/

/*
 * Exit is activated from the menu. This emits a delete event.
 */

void on_exit_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  gtk_signal_emit_by_name(GTK_OBJECT(mainwindow),"delete_event");
}

/*
 * Open is selected, just show the open dialog.
 */

void on_open_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  GtkWidget *opendialog;

  opendialog = create_opendialog();
  gtk_widget_show(opendialog);
}

/*
 * After close is selected, a dialog will show up, asking if you really want
 * to close the connection.
 */

void on_close_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  GtkWidget *closedialog, *closelabel;
  GString *labeltext;

  closedialog = create_closedialog ();
  closelabel = lookup_widget (closedialog, "closelabel");
  labeltext = g_string_new("Close connection to ");
  g_string_append(labeltext, cvhost);
  g_string_append(labeltext," ?");
  gtk_label_set_text(GTK_LABEL(closelabel), labeltext->str);
  g_string_free(labeltext, TRUE);
  gtk_widget_set_sensitive(mainwindow, 0);
  gtk_widget_show(closedialog);
}

/*
 * Activate preferences dialog.
 */
 
 
void on_preferences_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  preferencesdialog = create_preferencesdialog();
  gtk_widget_show(preferencesdialog);
}

/*
 * Activate the about window.
 */

void on_about_activate(GtkMenuItem *menuitem, gpointer user_data)
{
  GtkWidget *aboutdialog;

  aboutdialog = create_aboutdialog();
  gtk_widget_show(aboutdialog);
}

/******************************* OPEN DIALOG *********************************/

/*
 * When the open dialog appears, memory is allocated for the host and port
 * to connect to. The combo widgets are filled with their history.
 */

void on_opendialog_show(GtkWidget *widget, gpointer user_data)
{
  GtkWidget *opendialog, *hostnamecombo, *portcombo;
  
  if (!cvhost) cvhost = g_malloc0(100*sizeof(gchar));
  if (!cvport) cvport = g_malloc0(20*sizeof(gchar));
  opendialog = gtk_widget_get_toplevel(widget);
  hostnamecombo = lookup_widget(opendialog, "hostnamecombo");
  portcombo = lookup_widget(opendialog, "portcombo");
  if (hostnamehistory) gtk_combo_set_popdown_strings(GTK_COMBO(hostnamecombo),
    hostnamehistory);
  if (porthistory) gtk_combo_set_popdown_strings(GTK_COMBO(portcombo),
    porthistory);
  gtk_widget_set_sensitive(mainwindow, 0);
}

/*
 * When OK is clicked in the open dialog, the dialog is destroyed and the
 * actual connecting should start. If nothing is entered, values default to
 * 'localhost' and '3600'. Hostname and port are saved to a linked list,
 * up to a maximum of {CONNECT|PORT}HISTORY.
 * If a host or portname is selected from the combobox and it is not the first
 * one, it is moved to up. Before we actually connect, we do some cleanups.
 */

void on_openOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *opendialog, *hostnamecombo, *portcombo, *hostnamecomboentry,
    *portcomboentry;
  GList *node;

  opendialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  hostnamecombo = lookup_widget(opendialog, "hostnamecombo");
  portcombo = lookup_widget(opendialog, "portcombo");
  hostnamecomboentry = GTK_COMBO(hostnamecombo)->entry;
  portcomboentry = GTK_COMBO(portcombo)->entry;
  strcpy(cvhost, gtk_editable_get_chars(GTK_EDITABLE(hostnamecomboentry), 0,
    -1));
  strcpy(cvport, gtk_editable_get_chars(GTK_EDITABLE(portcomboentry), 0, -1));
  if (!strcmp(cvhost, "")) strcpy(cvhost, "localhost");
  if (!strcmp(cvport, "")) strcpy(cvport, "3600");
  node = g_list_find_custom(hostnamehistory, g_strdup(cvhost),
    (GCompareFunc)g_strncasecmp);
  if (!node) hostnamehistory = g_list_prepend(hostnamehistory,
    g_strdup(cvhost));
  else
  {
    if (g_list_position(hostnamehistory, node) != 0)
    {
      hostnamehistory = g_list_remove_link(hostnamehistory, node);
      hostnamehistory = g_list_prepend(hostnamehistory, g_strdup(cvhost));
    }
  }
  if (g_list_length(hostnamehistory) > HOSTNAMEHISTORY)
    hostnamehistory = g_list_remove(hostnamehistory,
      g_list_last(hostnamehistory)->data);
  node = g_list_find_custom(porthistory, g_strdup(cvport),
    (GCompareFunc)g_strncasecmp);
  if (!node) porthistory = g_list_prepend(porthistory, g_strdup(cvport));
  else
  {
    if (g_list_position(porthistory, node) != 0)
    {
      porthistory = g_list_remove_link(porthistory, node);
      porthistory = g_list_prepend(porthistory, g_strdup(cvport));
    }
  }
  if (g_list_length(porthistory) > PORTHISTORY)
    porthistory = g_list_remove(porthistory, g_list_last(porthistory)->data);
  if (rxmonitor) gdk_input_remove(rxmonitor);
  if (connected) connected = FALSE;
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
  cvconnect();
}

/*
 * Cancel is clicked, just destroy the dialog window.
 */

void on_opencancelbutton_clicked(GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET (button)));
}

/*
 * When the dialog is destroyed, the tx widget should get focus.
 */

void on_opendialog_destroy(GtkObject *object, gpointer user_data)
{
  GtkWidget *mainentry;
  
  gtk_widget_set_sensitive(mainwindow, 1);
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);
}

/***************************** CLOSE DIALOG **********************************/

/*
 * If OK is clicked on the close dialog, the socket is closed. I know this is
 * not nice to the server, I get a 'link failed'. The server wrongly assumes
 * that the link is down. Sending '/quit' won't work here, because the other
 * end could be down and I can wait for ages until the connection closes.
 * Maybe there exists some kind of socket timeout?
 */

void on_closeOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  cvdisconnect();
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 *  Just destroy the close dialog when cancel is clicked.
 */

void on_closecancelbutton_clicked(GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * When the dialog is destroyed, the tx widget should get focus.
 */

void on_closedialog_destroy(GtkObject *object, gpointer user_data)
{
  GtkWidget *mainentry;

  gtk_widget_set_sensitive(mainwindow, 1);
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);
}

/**************************** PREFERENCES DIALOG *****************************/

/*
 * Set the title of the preferences window and check or uncheck autologin and 
 * logging boxes. The preferences struct is checked to see if the name and 
 * channel strings are assigned. If so, the entries will be filled in. The 
 * background button gets the same color as the main text window and the font
 * entry is filled.
 */
 
void on_preferencesdialog_show(GtkWidget *widget, gpointer user_data)
{
  GtkStyle *style;
  GtkWidget *loggingcheckbutton, *autologincheckbutton, *autologinnamelabel, 
    *autologinchannellabel, *autologinnameentry, *autologinchannelentry,
    *backgroundcolorbutton, *fontentry, *menucheckbutton,
    *statusbarcheckbutton, *scrollbarcheckbutton;

  loggingcheckbutton = lookup_widget(widget, "loggingcheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(loggingcheckbutton), 
    preferences.logging);
  autologincheckbutton = lookup_widget(widget, "autologincheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(autologincheckbutton), 
    preferences.autologin);
  autologinnameentry = lookup_widget(widget, "autologinnameentry");
  autologinchannelentry = lookup_widget(widget, "autologinchannelentry");
  if (preferences.name)
    gtk_entry_set_text(GTK_ENTRY(autologinnameentry), preferences.name->str);
  if (preferences.channel)
    gtk_entry_set_text(GTK_ENTRY(autologinchannelentry), 
      preferences.channel->str);
  if (!preferences.autologin)
  {
    autologinnamelabel = lookup_widget(widget, "autologinnamelabel");
    autologinchannellabel = lookup_widget(widget, "autologinchannellabel");
    gtk_widget_set_sensitive(autologinnamelabel, 0);
    gtk_widget_set_sensitive(autologinchannellabel, 0);
    gtk_editable_set_editable(GTK_EDITABLE(autologinnameentry), FALSE);
    gtk_editable_set_editable(GTK_EDITABLE(autologinchannelentry), FALSE);
  }
  menucheckbutton = lookup_widget(widget, "menucheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(menucheckbutton), 
    preferences.menu);
  statusbarcheckbutton = lookup_widget(widget, "statusbarcheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(statusbarcheckbutton), 
    preferences.statusbar);
  scrollbarcheckbutton = lookup_widget(widget, "scrollbarcheckbutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scrollbarcheckbutton), 
    preferences.scrollbar);
  backgroundcolorbutton = lookup_widget(widget, "backgroundcolorbutton");
  style = gtk_style_new();
  style->bg[GTK_STATE_NORMAL] = preferences.background;
  style->bg[GTK_STATE_PRELIGHT] = preferences.background;
  gtk_widget_set_style(backgroundcolorbutton, style);
  fontentry = lookup_widget(widget, "fontentry");
  gtk_entry_set_text(GTK_ENTRY(fontentry), preferences.font->str);
  gtk_widget_set_sensitive(mainwindow, 0);
}

/*
 * The autologin checkbutton is toggled, we have to update the visibility and 
 * the editability of the name and channel labels and entries.
 */

void on_autologincheckbutton_toggled(GtkToggleButton *togglebutton,
  gpointer user_data)
{
  GtkWidget *autologinnamelabel, *autologinchannellabel, *autologinnameentry,
    *autologinchannelentry;
  gboolean autologincheckbuttonstatus;

  autologincheckbuttonstatus = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(togglebutton));
  preferencesdialog = gtk_widget_get_toplevel(GTK_WIDGET(togglebutton));
  autologinnamelabel = lookup_widget(preferencesdialog, "autologinnamelabel");
  autologinchannellabel = lookup_widget(preferencesdialog, 
    "autologinchannellabel");
  autologinnameentry = lookup_widget(preferencesdialog, "autologinnameentry");
  autologinchannelentry = lookup_widget(preferencesdialog, 
    "autologinchannelentry");
  if (autologincheckbuttonstatus)
  {
    gtk_widget_set_sensitive(autologinnamelabel, 1);
    gtk_widget_set_sensitive(autologinchannellabel, 1);
    gtk_widget_set_sensitive(autologinnameentry, 1);
    gtk_widget_set_sensitive(autologinchannelentry, 1);
    gtk_editable_set_editable(GTK_EDITABLE(autologinnameentry), TRUE);
    gtk_editable_set_editable(GTK_EDITABLE(autologinchannelentry), TRUE);
  }
  else
  {
    gtk_widget_set_sensitive(autologinnamelabel, 0);
    gtk_widget_set_sensitive(autologinchannellabel, 0);
    gtk_widget_set_sensitive(autologinnameentry, 0);
    gtk_widget_set_sensitive(autologinchannelentry, 0);
    gtk_editable_set_editable(GTK_EDITABLE(autologinnameentry), FALSE);
    gtk_editable_set_editable(GTK_EDITABLE(autologinchannelentry), FALSE);
  }
}

/*
 * We want to change the background color of the main window, so we look up the
 * color and apply this to the color selection dialog.
 */

void on_backgroundcolorbutton_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *colorselectiondialog;
  gdouble rgb[3];

  colorselectiondialog = create_colorselectiondialog();
  gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(colorselectiondialog )->
    help_button);
  rgb[0] = preferences.background.red / 65535.0;
  rgb[1] = preferences.background.green / 65535.0;
  rgb[2] = preferences.background.blue / 65535.0;
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(
    GTK_COLOR_SELECTION_DIALOG(colorselectiondialog)->colorsel), rgb);
  gtk_widget_set_sensitive(preferencesdialog, 0);
  gtk_widget_show(colorselectiondialog);
}

/*
 * Here we change the font of the main text window. The font is looked up in
 * the preferences struct and applied to the dialog.
 */

void on_fontbutton_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *fontselectiondialog;

  fontselectiondialog = create_fontselectiondialog();
  gtk_font_selection_dialog_set_preview_text(GTK_FONT_SELECTION_DIALOG
    (fontselectiondialog), "How about this font?");
  gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG
    (fontselectiondialog), preferences.font->str);
  gtk_widget_set_sensitive(preferencesdialog, 0);
  gtk_widget_show(fontselectiondialog);
}

/*
 * Save setting, update preferences struct and destroy the dialog.
 * If logging is checked and we are connected, the log file should be created.
 * If logging is unchecked and there is a logfile, we should close it.
 * The autologin settings are applied and the background is set.
 */

void on_preferencesOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *loggingcheckbutton, *autologincheckbutton, *autologinnameentry,
    *autologinchannelentry, *maintext, *menucheckbutton, *statusbarcheckbutton,
    *scrollbarcheckbutton, *mainscrolledwindow;
  gboolean loggingcheckbuttonstatus, autologincheckbuttonstatus,
    menucheckbuttonstatus, statusbarcheckbuttonstatus,
    scrollbarcheckbuttonstatus;
  GtkStyle *maintextstyle;

  loggingcheckbutton = lookup_widget(preferencesdialog, "loggingcheckbutton");
  loggingcheckbuttonstatus = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(loggingcheckbutton));
  preferences.logging = loggingcheckbuttonstatus;
  autologincheckbutton = lookup_widget(preferencesdialog, 
    "autologincheckbutton");
  autologincheckbuttonstatus = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(autologincheckbutton));
  preferences.autologin = autologincheckbuttonstatus;
  if (preferences.autologin)
  {
    autologinnameentry = lookup_widget(preferencesdialog, 
      "autologinnameentry");
    autologinchannelentry = lookup_widget(preferencesdialog, 
      "autologinchannelentry");
    preferences.name = g_string_new(gtk_editable_get_chars(GTK_EDITABLE
      (autologinnameentry), 0, -1));
    preferences.channel = g_string_new(gtk_editable_get_chars(GTK_EDITABLE
      (autologinchannelentry), 0, -1));
  }
  menucheckbutton = lookup_widget(preferencesdialog, "menucheckbutton");
  menucheckbuttonstatus = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(menucheckbutton));
  widget_visible("mainmenubar", menucheckbuttonstatus);
  preferences.menu = menucheckbuttonstatus;
  statusbarcheckbutton = lookup_widget(preferencesdialog,
    "statusbarcheckbutton");
  statusbarcheckbuttonstatus = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(statusbarcheckbutton));
  widget_visible("mainstatusbar", statusbarcheckbuttonstatus);
  preferences.statusbar = statusbarcheckbuttonstatus;
  scrollbarcheckbutton = lookup_widget(preferencesdialog,
    "scrollbarcheckbutton");
  scrollbarcheckbuttonstatus = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(scrollbarcheckbutton));
  preferences.scrollbar = scrollbarcheckbuttonstatus;
  mainscrolledwindow = lookup_widget(mainwindow, "mainscrolledwindow");
  if (scrollbarcheckbuttonstatus)
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainscrolledwindow), 
      GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  else
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainscrolledwindow), 
      GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  savepreferences();
  if (preferences.logging && connected)
  {
    logfilename = g_strconcat(preferencesdir, "/", cvhost, ".log", NULL);
    if ((logfp = fopen(logfilename, "w")) == NULL) 
      g_error("Creating %s for writing.", logfilename);
  }
  else 
  { 
    if (!preferences.logging && connected) fclose(logfp);
  }
  maintext = lookup_widget(mainwindow, "maintext");
  maintextstyle = gtk_style_new();
  maintextstyle->base[GTK_STATE_NORMAL] = preferences.background;
  maintextstyle->base[GTK_STATE_INSENSITIVE] = preferences.background;
  gtk_widget_set_style(maintext, maintextstyle);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET (button)));
}

/*
 * Just destroy the preferences dialog when cancel is clicked.
 */

void on_preferencescancelbutton_clicked(GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * When the preferences window is deleted the tx widget gets focus.
 */
 
void on_preferencesdialog_destroy(GtkObject *object, gpointer user_data)
{
  GtkWidget *mainentry;

  gtk_widget_set_sensitive(mainwindow, 1);
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);
}

/****************************** ABOUT DIALOG *********************************/

/*
 * Update the about dialog with some information before it is shown.
 * Title of about window is set here too.
 */

void on_aboutdialog_show(GtkWidget *widget, gpointer user_data)
{
  GString *info;
  GtkWidget *aboutcreditslabel, *aboutlicenselabel, *abouthelptext;

  aboutcreditslabel = lookup_widget(widget, "aboutcreditslabel");
  abouthelptext = lookup_widget(widget, "abouthelptext");
  aboutlicenselabel = lookup_widget(widget, "aboutlicenselabel");
  info = g_string_new("\n");
  g_string_append(info, PACKAGE);
  g_string_append(info, " version ");
  g_string_append(info, VERSION);
  g_string_append(info, "\nCopyright (C) 2000-2001 Joop Stakenborg\n");
  g_string_append(info, "E-mail: <pa4tu@amsat.org>\n");
  g_string_append(info, "http://xconvers.sourceforge.net\n\n");
  gtk_label_set_text(GTK_LABEL(aboutcreditslabel), info->str);
  g_string_free(info, TRUE);
  info = g_string_new("");
  g_string_append(info, "PgUp: Scroll the receive window up.\n");
  g_string_append(info, "PgDn: Scroll the receive window down.\n");
  g_string_append(info, "Up Arrow: Recall previously transmitted text.\n");
  g_string_append(info, "Dn Arrow: Recall next transmitted text.\n\n");
  g_string_append(info, "Alt+F: File menu (exit program).\n");
  g_string_append(info, "Alt+H: Host menu (open/close a connection).\n");
  g_string_append(info, "Alt+S: Settings menu (preferences).\n");
  g_string_append(info, "Alt+P: Help menu (credits and help).\n");
  gtk_text_insert(GTK_TEXT(abouthelptext), NULL, NULL, NULL, info->str, -1);
  g_string_free(info, TRUE);
  info = g_string_new("\n");
  g_string_append(info, "Xconvers comes with ABSOLUTELY NO WARRANTY.\n");
  g_string_append(info, "This is free software, and you are welcome to\n"); 
  g_string_append(info, "redistribute it under certain conditions.\n");
  g_string_append(info, "For details see the GNU General Public License.\n");
  gtk_label_set_text(GTK_LABEL(aboutlicenselabel), info->str);
  g_string_free(info, TRUE);
}

/*
 * Clicking OK just destroys this widget and gives focus to the tx widget.
 */

void on_aboutOKbutton_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *mainentry;
  
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
  mainentry = lookup_widget(mainwindow, "mainentry");
  gtk_widget_grab_focus(mainentry);  
}

/************************** COLOR SELECTION DIALOG ***************************/

/*
 * We have selected a color, so now we can set the color of the button on the 
 * preferences dialog.
 */

void on_colorok_button_clicked(GtkButton *button, gpointer user_data)
{
  GtkWidget *colorselectiondialog, *backgroundcolorbutton;
  GtkStyle *backgroundcolorbuttonstyle;
  gdouble rgb[3];
  
  backgroundcolorbuttonstyle = gtk_style_new();
  colorselectiondialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG
    (colorselectiondialog)->colorsel), rgb);
  preferences.background.red = rgb[0] * 65535.0;
  preferences.background.green = rgb[1] * 65535.0;
  preferences.background.blue = rgb[2] * 65535.0;
  backgroundcolorbuttonstyle->bg[GTK_STATE_NORMAL] = preferences.background;
  backgroundcolorbuttonstyle->bg[GTK_STATE_PRELIGHT] = preferences.background;
  backgroundcolorbutton = lookup_widget(preferencesdialog,
    "backgroundcolorbutton");
  gtk_widget_set_style(backgroundcolorbutton, backgroundcolorbuttonstyle);
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * Just destroy the dialog when cancel is clicked.
 */

void on_colorcancel_button_clicked (GtkButton *button, gpointer user_data)
{
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/************************* FONT SELECTION DIALOG *****************************/

/*
 * A font is selected. Update the preferences struct and change the text font
 * for the main window.
 */

void on_fontok_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *fontselectiondialog, *fontentry;
  
  fontselectiondialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  preferences.font = g_string_new(gtk_font_selection_dialog_get_font_name
    (GTK_FONT_SELECTION_DIALOG(fontselectiondialog)));
  fontentry = lookup_widget(preferencesdialog, "fontentry");
  gtk_entry_set_text(GTK_ENTRY(fontentry), preferences.font->str);
  textfont = gdk_font_load(preferences.font->str);
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/*
 * Just destroy the dialog when cancel is clicked.
 */

void on_fontcancel_button_clicked (GtkButton *button, gpointer user_data)
{
  gtk_widget_set_sensitive(preferencesdialog, 1);
  gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}
