/*
 * xconvers - GTK+ convers client for amateur radio
 * Copyright (C) 2000-2001 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * color.c - private functions for looking up and displaying colors.
 */
 
#include <gtk/gtk.h>
#include <stdlib.h>
#include "support.h"
#include "color.h"
#include "net.h"
#include "types.h"

extern GtkWidget *mainwindow;
extern GdkColor white, grey, red, pink, color[10];
extern GdkFont *textfont;
extern preferencestype preferences;
gchar *savedline;
gint previousmessage = 0, usercolor = 0;
GHashTable *callsigns = NULL;

/*
 * Different messages are displayed in different colors here. This all depends
 * on the first couple of characters on a line. An incoming message, which can
 * have multiple lines, is first split into lines with g_strsplit, so the lines
 * can be checked for a color. When a line starts with '<', it definitely is a
 * user message. We try to extract the user's callsign with g_strdelimit and 
 * g_strsplit. Finally we call getcolor, to get the color associated with the 
 * user.
 */

void maintext_add(gchar *msg, gint messagetype)
{
  GtkWidget *maintext;
  gchar **rxsplit = NULL, **linesplit = NULL;
  gchar *delimitedline = NULL, *insert = NULL, *callsign = NULL;
  gchar *tmpstr = NULL;
  gint i, colorindex = 0, lines = 0, crs = 0, numbytes = 0;
  gint rxtype = DEFAULT_MESSAGE;

  numbytes = strlen(msg);
  if (msg[0] == '\0') return;
  maintext = lookup_widget (mainwindow, "maintext");
  switch (messagetype)
  {
    case MESSAGE_RX:
      rxsplit = g_strsplit(msg, "\n", 0);
      for (i = 0; i < numbytes; i++) if (msg[i] == '\n') crs++;
      for (lines = 0; lines <= crs + 1; lines++)
        if (!rxsplit[lines]) break;
      if (lines > crs) tmpstr = g_strdup(rxsplit[lines - 1]);
        else tmpstr = NULL;
      if (savedline)
      {
        rxsplit[0] = g_strconcat(savedline, rxsplit[0], NULL);
        savedline = NULL;
      }
      for (i = 0; i < crs; i++)
      {
        insert = g_strdup_printf("%s\n", rxsplit[i]);
        if (g_strncasecmp(insert, "*", 1) == 0)
        {
          if (insert[1] == '*') rxtype = STATUS_MESSAGE;
          else rxtype = DEFAULT_MESSAGE;
        }
        if (g_strncasecmp(insert, "<", 1) == 0)
        {
          if (insert[1] == '*') rxtype = PRIVATE_MESSAGE;
          else rxtype = USER_MESSAGE;
        }
        if (g_strncasecmp(insert, " ", 1) == 0) rxtype = previousmessage;
        switch (rxtype)
        {
          case STATUS_MESSAGE:
            gtk_text_insert(GTK_TEXT(maintext), textfont, &red, NULL, insert, 
              -1);
            previousmessage = STATUS_MESSAGE;
          break;
          case USER_MESSAGE:
            if (g_strncasecmp(insert, " ", 1) != 0)
            {
              delimitedline = g_strdelimit(rxsplit[i], "<>", '#');
              linesplit = g_strsplit(delimitedline, "#", 3);
              callsign = g_strstrip(linesplit[1]);
              colorindex = getcolor(callsign);
              g_strfreev(linesplit);
            }
            gtk_text_insert(GTK_TEXT(maintext), textfont, &color[colorindex], 
              NULL, insert, -1);
            previousmessage = USER_MESSAGE;
          break;
          case PRIVATE_MESSAGE:
            gtk_text_insert(GTK_TEXT(maintext), textfont, &pink, NULL, insert,
              -1);
            previousmessage = PRIVATE_MESSAGE;
          break;
          case UNKNOWN_MESSAGE:
          break;
          case DEFAULT_MESSAGE:
            gtk_text_insert(GTK_TEXT(maintext), textfont, &grey, NULL, insert,
              -1);
            previousmessage = DEFAULT_MESSAGE;
          break;
          default:
          break;
        }
      }
      free(insert);
      g_strfreev(rxsplit);
      if (tmpstr) savedline = g_strdup(tmpstr); else savedline = NULL;
    break;
    case MESSAGE_TX:
      gtk_text_insert(GTK_TEXT(maintext), textfont, &white, NULL, msg, -1);
    break;
    default:
    break;
  }
}

/*
 * Here the color associated with a callsign is looked up. Information is 
 * stored in a hash table. This will ensure fast lookups. Items are added using
 * a key, which is calculated from the callsign. When a call is not found, the
 * color counter is incremented.
 */

gint getcolor(gchar *color)
{
  gchar *call = g_strdup(color);
  gint i;

  if (g_hash_table_lookup(callsigns, call) == NULL)
  {
    usercolor++;
    g_hash_table_insert(callsigns, g_strdup(call), GINT_TO_POINTER(usercolor));
    i = usercolor;
  }
  else
    i = GPOINTER_TO_INT(g_hash_table_lookup(callsigns, call));
  return((i-1) % 10);
}

/*
 * The color which is looked up in the xconvers preferences file is put in the
 * right format for XParseColor, so it can be processed by gdk_color_parse.
 */

gchar *color_parse(gchar value[100])
{
  gchar **valuesplit = NULL;
  gchar *rgbsyntax;
  
  valuesplit = g_strsplit(value, ",", 3);
  rgbsyntax = g_strconcat("rgb:", valuesplit[0], "/", valuesplit[1], "/",
    valuesplit[2], NULL);
  g_strfreev(valuesplit);
  return (rgbsyntax);
}