/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>
#include <parted/disk_dos.h>
#include <parted/disk_loop.h>
#include <parted/disk_pc98.h>
#include <parted/disk_mac.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <unistd.h>
#include <asm/page.h>
#include <string.h>

#define NTFS_SIGNATURE	"NTFS"

static int ntfs_probe (const PedGeometry* geom);
static int ntfs_clobber (PedGeometry* geom);
static int ntfs_set_system (const PedFileSystem* fs, PedPartition* part,
			    const PedDiskType* disk_type);

static PedFileSystemOps ntfs_ops = {
	probe:		ntfs_probe,
	clobber:	ntfs_clobber,
	open:		NULL,
	create:		NULL,
	close:		NULL,
	check:		NULL,
	copy:		NULL,
	resize:		NULL,
	get_resize_constraint:	NULL,
	set_system:	ntfs_set_system
};

static PedFileSystemType ntfs_type = {
	next:	NULL,
	ops:	&ntfs_ops,
	name:	"ntfs"
};

void
ped_file_system_ntfs_init ()
{
	ped_file_system_type_register (&ntfs_type);
}

void
ped_file_system_ntfs_done ()
{
	ped_file_system_type_unregister (&ntfs_type);
}

static int
ntfs_probe (const PedGeometry* geom)
{
	char	buf[512];

	if (!ped_geometry_read (geom, buf, 0, 1))
		return 0;
	return strncmp (NTFS_SIGNATURE, buf + 3, strlen (NTFS_SIGNATURE)) == 0;
}

static int
ntfs_clobber (PedGeometry* geom)
{
	char	buf[512];

	memset (buf, 0, 512);
	return ped_geometry_write (geom, buf, 0, 1);
}

static int
ntfs_set_system (const PedFileSystem* fs, PedPartition* part,
		 const PedDiskType* disk_type)
{
	if (strcmp (disk_type->name, LOOP_NAME) == 0)
		return 1;

	if (strcmp (disk_type->name, DOS_NAME) == 0) {
		DosPartitionData*	dos_data = part->disk_specific;

		PED_ASSERT (dos_data != NULL, return 0);

		if (dos_data->raid) {
			dos_data->system = 0xfd;
			return 1;
		}
		if (dos_data->lvm) {
			dos_data->system = 0x8e;
			return 1;
		}

		if (dos_data->hidden) {
			ped_exception_throw (
				PED_EXCEPTION_ERROR,
			       	PED_EXCEPTION_CANCEL,
				_("NTFS partitions can't be hidden on msdos "
				  "disk labels."));
			return 0;
		}

		dos_data->system = 0x7;
		return 1;
	}

	if (strcmp (disk_type->name, PC98_NAME) == 0) {
		PC98PartitionData*	pc98_data = part->disk_specific;

		PED_ASSERT (pc98_data != NULL, return 0);

		pc98_data->system = 0x20b1;
		return 1;
	}

	if (strcmp (disk_type->name, MAC_NAME) == 0) {
		MacPartitionData*	mac_data = part->disk_specific;

		PED_ASSERT (mac_data != NULL, return 0);

		if (mac_data->is_boot)
			strcpy (mac_data->system_name, "Apple_Bootstrap");
		else
			strcpy (mac_data->system_name, "Apple_UNIX_SVR2");
		mac_data->status = 0x33;
		return 1;
	}

	return 0;
}
