/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>
#include <parted/disk_dos.h>
#include <parted/disk_loop.h>
#include <parted/disk_pc98.h>
#include <parted/disk_mac.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <unistd.h>
#include <asm/page.h>
#include <string.h>

#define REISERFS_SIGNATURE	"ReIsErFs"
#define REISER2FS_SIGNATURE	"ReIsEr2Fs"

static PedSector reiser_super_offset[] = {
	128,
	16,			/* versions 3.5 - 3.5.11 */
	-1
};

static int reiserfs_probe (const PedGeometry* geom);
static int reiserfs_clobber (PedGeometry* geom);
static int reiserfs_set_system (const PedFileSystem* fs, PedPartition* part,
				const PedDiskType* disk_type);

static PedFileSystemOps reiserfs_ops = {
	probe:		reiserfs_probe,
	clobber:	reiserfs_clobber,
	open:		NULL,
	create:		NULL,
	close:		NULL,
	check:		NULL,
	copy:		NULL,
	resize:		NULL,
	get_resize_constraint:	NULL,
	set_system:	reiserfs_set_system
};

static PedFileSystemType reiserfs_type = {
	next:	NULL,
	ops:	&reiserfs_ops,
	name:	"reiserfs"
};

void
ped_file_system_reiserfs_init ()
{
	ped_file_system_type_register (&reiserfs_type);
}

void
ped_file_system_reiserfs_done ()
{
	ped_file_system_type_unregister (&reiserfs_type);
}

static int
reiserfs_probe (const PedGeometry* geom)
{
	char	buf[512];
	int	i;

	for (i=0; reiser_super_offset[i] != -1; i++) {
		if (reiser_super_offset[i] >= geom->length)
			continue;
		if (!ped_geometry_read (geom, buf, reiser_super_offset[i], 1))
			continue;
		if (strncmp (REISERFS_SIGNATURE, buf + 52,
			     strlen (REISERFS_SIGNATURE)) == 0)
			return 1;
		if (strncmp (REISER2FS_SIGNATURE, buf + 52,
			     strlen (REISER2FS_SIGNATURE)) == 0)
			return 1;
	}
	return 0;
}

static int
reiserfs_clobber (PedGeometry* geom)
{
	char	buf[512];
	int	i;

	memset (buf, 0, 512);
	for (i=0; reiser_super_offset[i] != -1; i++) {
		if (reiser_super_offset[i] >= geom->length)
			continue;
		if (!ped_geometry_write (geom, buf, reiser_super_offset[i], 1))
			return 0;
	}
	return 1;
}

static int
reiserfs_set_system (const PedFileSystem* fs, PedPartition* part,
		     const PedDiskType* disk_type)
{
	if (strcmp (disk_type->name, LOOP_NAME) == 0)
		return 1;

	if (strcmp (disk_type->name, DOS_NAME) == 0) {
		DosPartitionData*	dos_data = part->disk_specific;

		PED_ASSERT (dos_data != NULL, return 0);

		if (dos_data->raid) {
			dos_data->system = 0xfd;
			return 1;
		}
		if (dos_data->lvm) {
			dos_data->system = 0x8e;
			return 1;
		}

		if (dos_data->hidden) {
			ped_exception_throw (
				PED_EXCEPTION_ERROR,
			       	PED_EXCEPTION_CANCEL,
				_("Reiserfs partitions can't be hidden on "
				  "msdos disk labels."));
			return 0;
		}

		dos_data->system = 0x83;
		return 1;
	}

	if (strcmp (disk_type->name, PC98_NAME) == 0) {
		PC98PartitionData*	pc98_data = part->disk_specific;

		PED_ASSERT (pc98_data != NULL, return 0);

		if (!pc98_data->boot) {
			ped_exception_throw (
				PED_EXCEPTION_ERROR,
				PED_EXCEPTION_FIX,
				_("Reiserfs partitions must be bootable on "
				  "pc98 disk labels."));
			pc98_data->boot = 1;
		}

		pc98_data->system = 0xa0e2;
		return 1;
	}

	if (strcmp (disk_type->name, MAC_NAME) == 0) {
		MacPartitionData*	mac_data = part->disk_specific;

		PED_ASSERT (mac_data != NULL, return 0);

		if (mac_data->is_boot)
			strcpy (mac_data->system_name, "Apple_Bootstrap");
		else
			strcpy (mac_data->system_name, "Apple_UNIX_SVR2");
		mac_data->status = 0x33;
		return 1;
	}

	return 0;
}
