/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include "gtm_applet.h"
#include "dialogs.h"

/* Variables used in the selection of the presume protocol */
#define NUM_PRESUME_PROTOCOLS 3

static Protocol presume_protocol[NUM_PRESUME_PROTOCOLS] = {
    PROTOCOL_NONE,
    PROTOCOL_HTTP,
    PROTOCOL_FTP
};

static gchar *presume_protocol_label[NUM_PRESUME_PROTOCOLS] = {
    N_ ("None"),
    N_ ("HTTP"),
    N_ ("FTP")
};


/* Structure that keeps the widgets containing the current properties */
typedef struct 
{
    GtkWidget *pbox;                            /* GNOME Property Box */
    GtkWidget *confirm_dnd_url;                 /* Toggle button */
    gint binding_button1_click;
    GtkWidget *presume_protocol[NUM_PRESUME_PROTOCOLS]; /* Toggle button */
} PropDialogData;

static PropDialogData *prop_dialog_data = NULL;

static void
apply_prop_download (PropDialogData *prop)
{
    guint i;

    /* Presume protocol */
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++) {
        if (gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (prop->presume_protocol[i]))) {
            gtm_applet_prop.presume_protocol = presume_protocol[i];
            break;
        }
    }
}

static void
apply_prop_confirmations (PropDialogData *prop)
{
    /* Confirm drag-n-drop URL */
    gtm_applet_prop.confirm_dnd_url =
        gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (prop->confirm_dnd_url));
}

static void
apply_prop_bindings (PropDialogData *prop)
{
    /* Set new button 1 click binding */
    gtm_applet_prop.binding_button1_click = prop->binding_button1_click;
}

static void
apply_prop (GtkWidget *widget, gint page_num, PropDialogData *prop)
{
    /* If it's not an absolute and we don't use that */
    if (page_num == -1)
        return;

    /* This function is called for each page changed so we need to apply props
     * for each page individually */
    switch (page_num) {
        case 0: /* Download */
            apply_prop_download (prop);
            break;

        case 1: /* Confirmations */
            apply_prop_confirmations (prop);
            break;

        case 2: /* Bindings */
            apply_prop_bindings (prop);
            break;
    }

    /* Save a global default configuration */
    gnome_config_push_prefix ("gtm_applet/Default/");
    gnome_config_set_bool ("confirm_dnd_url",
                           gtm_applet_prop.confirm_dnd_url);
    gnome_config_set_int ("binding_button1_click",
                          gtm_applet_prop.binding_button1_click);
    gnome_config_set_int ("presume_protocol",
                          gtm_applet_prop.presume_protocol);
    gnome_config_pop_prefix ();
}

static void
destroy_propbox (GtkWidget *widget, PropDialogData **prop)
{
    gtk_widget_destroy ((*prop)->pbox);
    g_free (*prop);
    *prop = NULL;
}

static void
prop_modified_cb (GtkWidget *widget, PropDialogData *prop)
{
    /* Tell to property box that data changed */
    gnome_property_box_changed (GNOME_PROPERTY_BOX (prop->pbox));
}

static void
binding_click_changed_cb (GtkWidget *widget, gint index)
{
    prop_dialog_data->binding_button1_click = index;
    gnome_property_box_changed (GNOME_PROPERTY_BOX (prop_dialog_data->pbox));
}

static GtkWidget *
create_download_page (PropDialogData *prop)
{
    GtkWidget *vbox, *hbox, *label;
    GSList *group;
    guint i;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* Toolbar style */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("Presume Protocol:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    group = NULL;
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++) {
        prop->presume_protocol[i] = 
            gtk_radio_button_new_with_label (group,
                                             _ (presume_protocol_label[i]));
        gtk_box_pack_start (GTK_BOX (hbox), prop->presume_protocol[i], FALSE, 
                            FALSE, 0);
        group = 
            gtk_radio_button_group (GTK_RADIO_BUTTON (
                prop->presume_protocol[i]));
    }    

    /* Set current properties */
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++) {
        if (gtm_applet_prop.presume_protocol == presume_protocol[i]) {
            gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (prop->presume_protocol[i]), TRUE);
            break;
        }
    }

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    for (i = 0; i < NUM_PRESUME_PROTOCOLS; i++)
        gtk_signal_connect (GTK_OBJECT (prop->presume_protocol[i]), "clicked",
                            GTK_SIGNAL_FUNC (prop_modified_cb),
                            (gpointer) prop);

    return vbox;
}

static GtkWidget *
create_confirmations_page (PropDialogData *prop)
{
    GtkWidget *vbox;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* Confirm drag-n-drop URL */
    prop->confirm_dnd_url = 
        gtk_check_button_new_with_label (_ (
            "Confirm drag and drop URL"));
    gtk_box_pack_start (GTK_BOX (vbox), prop->confirm_dnd_url, FALSE,
                        FALSE, 0);

    /* Set current properties */
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        prop->confirm_dnd_url), gtm_applet_prop.confirm_dnd_url);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (prop->confirm_dnd_url), "toggled",
                        GTK_SIGNAL_FUNC (prop_modified_cb),
                        (gpointer) prop);

    return vbox;
}

static GtkWidget *
create_bindings_page (PropDialogData *prop)
{
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *menu;
    GtkWidget *optionmenu;
    GtkWidget *menuitem;
    GtkWidget *label;
    gint index;
    static const char *applet_actions[] = { 
        N_ ("(none)"),
        N_ ("New URL"),
        N_ ("Open GTM"),
        NULL
    };

    /* Create vbox */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);
    
    /* Binding for button1-click */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    label = gtk_label_new (_ ("Button1 click:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    optionmenu = gtk_option_menu_new ();
    gtk_box_pack_start (GTK_BOX (hbox), optionmenu, FALSE, FALSE, 0);
    menu = gtk_menu_new();
    for (index = 0; applet_actions[index] != NULL; ++index) {
        menuitem = gtk_menu_item_new_with_label (applet_actions[index]);
        gtk_menu_append (GTK_MENU (menu), menuitem);
        gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                            GTK_SIGNAL_FUNC (binding_click_changed_cb),
                            (gpointer) index);
        gtk_widget_show (menuitem);
    }
    gtk_menu_set_active (GTK_MENU (menu), 
                         gtm_applet_prop.binding_button1_click);
    gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
                              menu);
    gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show (label);
    gtk_widget_show (optionmenu);
    
    return vbox;
}

/* Function to create and run the properties diaload box */
void
dialog_prop (void)
{
    GtkWidget *label;

    /* Only one properties dialog box is allowed */
    if (prop_dialog_data != NULL) {
        gdk_window_raise (GTK_WIDGET (
            GNOME_DIALOG (prop_dialog_data->pbox))->window);
        return;
    }

    /* Create the gnome property box and set the common signals */
    prop_dialog_data = g_new (PropDialogData, 1);
    prop_dialog_data->pbox = gnome_property_box_new ();
    gtk_window_set_title (GTK_WINDOW (prop_dialog_data->pbox),
                          _ ("Properties"));
    gtk_signal_connect (GTK_OBJECT (prop_dialog_data->pbox), "destroy",
                        GTK_SIGNAL_FUNC (destroy_propbox), 
                        (gpointer) &prop_dialog_data);
    gtk_signal_connect (GTK_OBJECT (prop_dialog_data->pbox), "apply",
                        GTK_SIGNAL_FUNC (apply_prop), 
                        (gpointer) prop_dialog_data);

    /* Create the notebook pages */
    label = gtk_label_new (_ ("Download"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (prop_dialog_data->pbox)->notebook), 
        create_download_page (prop_dialog_data), label);
    label = gtk_label_new (_ ("Confirmations"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (prop_dialog_data->pbox)->notebook), 
        create_confirmations_page (prop_dialog_data), label);
    label = gtk_label_new (_ ("Bindings"));
    gtk_notebook_append_page (
        GTK_NOTEBOOK (GNOME_PROPERTY_BOX (prop_dialog_data->pbox)->notebook),
        create_bindings_page (prop_dialog_data), label);

    /* We are ready to show it all */
    gtk_widget_show_all (prop_dialog_data->pbox);
}
